// Copyright (c) 2021 Kana LLC

//! Functions that do not fall into the general categories of checking, mutating, or generating a new sequence. Generally includes statistics and transformations.
//! # Examples
//! ``
//! use crate::bioutils::get::value;
//! use crate::bioutils::utils::new::random_dna;
//! use rand::rngs::ThreadRng;
//! use std::string::String;
//! use rand::seq::SliceRandom;
//! 
//! let mut rng = rand::thread_rng(); //create a random number generator
//! let dna = random_dna(4,rng);
//! let distance = dna.hamming_distance(b"AAAA");
//! println!("{:?}", distance);
//! ``

use crate::charsets::PERCENTAGE_RANGE;
use std::collections::HashMap;
use crate::charsets::iupac::*;
use crate::charsets::quality::*;
use rand::rngs::ThreadRng;
use rand::seq::SliceRandom;
use std::convert::TryInto;

pub trait ValueU8<T> {
    /// Returns the percent (0-100) of the quality u8 in bases (rounded) above the quality score supplied. Should be used when mapq scores are required.
    fn quality_percent_passing(&self, quality_score: &u8) -> usize;

    /// Returns the hamming distance of self and another seq.
    fn hamming_distance(&self, seq2: &T) -> u64;

    /// Returns the number of iterators greater than criteria. Used for calculating percents/numerators
    fn count_greater_than(&self, criteria:&u8)-> usize;

    /// Returns the number of occurrences of the mode
    fn count_mode(&self) -> usize;

    /// Returns the mode
    fn mode(&self) -> Option<&u8>;

    /// Returns the count of a specific u8
    fn count_xu8(&self, x: &u8) -> usize;
}

impl<T> ValueU8<T> for T
where
    for<'a> &'a T: IntoIterator<Item = &'a u8>, 
{
    /// Checks each quality u8 and returns the percent above (passing) the given u8
    fn quality_percent_passing(&self, quality_score: &u8)-> usize {
        percentage(self.count_greater_than(&quality_score), self.into_iter().count())
    }

    /// Checks the hamming distance between our item and a supplied item
    fn hamming_distance(&self, seq2: &T) -> u64 {
        let seq1=self;
        seq1.into_iter().zip(seq2).fold(0, |seq1, (seq2, c)| seq1 + (*seq2 ^ *c).count_ones() as u64)
    }

    /// Returns the number of iterations greater than the criteria
    fn count_greater_than(&self, criteria:&u8)-> usize {
        self.into_iter().filter(|s| s>=&criteria).count()
    }

    /// Returns the number of occurrences of the mode
    fn count_mode(&self) -> usize {
        let mode = self.mode().unwrap();
        self.into_iter().filter(|&q| q==mode).count()
    }

    /// Returns the mode
    fn mode(&self) -> Option<&u8> {
        let mut counts = HashMap::new();
        self.into_iter().max_by_key(|&s| {
            let count = counts.entry(s).or_insert(0);
            *count += 1;
            *count})
    }
    
    /// Returns the count of a specific u8
    fn count_xu8(&self, x: &u8) -> usize {
        self.into_iter().filter(|&q| q==x).count()
    }
}

/// Calculates percentage with usizes
pub fn percentage(numerator: usize, denominator: usize) -> usize {
    (100 * numerator + denominator / 2) / denominator
}

/// Validate a u8 is 0 to 100 and return a wrapped boolean
pub fn validate_percentage_u8(percent: &u8) -> Result<bool, &'static str> {
    match PERCENTAGE_RANGE.contains(percent){    
    true => Ok(true),
    false => Err("Please supply a percent (0-100, not fractional) as u8"),
    }
}

//TODO
// /// Validate if a quality score is phred33, phred64, etc. Example: validate_quality_score_u8(quality_score, phred33u8hashmap)
// pub fn validate_quality_score_u8(quality_score: , quality_charset: &str){};

// /// Get the percentage content in a u8 sequence

// /// Get the percentage content of a sequence in a u8 sequence

// // let x = b"ABCD";
// // let y = b"1010";
// // println!(b"BBDD");
// // println!(b"BBDD");

// // /// Take in a sequence string and create a vector of sequence strings with hamming distance 1 using the bases ACTG. Requires the sequence to be ACTGs, use replace if N.- or other symbols present.
// // // Example: AAAA -> CAAA GAAA TAAA ACAA AGAA ATAA etc.
// // pub fn nucleotide_set_hamming(nucl: String) -> Vec<String>  {
// //     let mut rng = rand::thread_rng();
// //     for base in 0 .. nucl.len() + 1 {
// //     let results: Vec<String> = Vec::new();
// //         for symbol in BASIC_DNA_U8.iter() {
// //             let (first, last) = symbol.split_at(base);
// //             let mut buffer = [0; 1];
// //             let result = symbol.encode_utf8(&mut buffer);
// //             results.push([first, result, last].concat());
// //         }
// //     results
// //     }
// // }
// // //-> Vec<String> 
// // // /// A function that returns the correction for the specified word.
// // // pub fn correct(&mut self, word: &str) -> String {
// // //     // A word in our word frequency map is already correct.
// // //     if self.n_words.contains_key(word) {
// // //         return word.to_string();
// // //     }

