//! # MiniGrep Crate
//!
//! `minigrep` is a app that like cmd `grep` in linux amd
//! you can use it everywhere
use std::env;
use std::error::Error;
use std::fs;

/// Run a minigrep server
///
/// # Examples
///
/// ```
/// let args = env::args();
/// let config = Config::new(args).unwrap_or_else(|err| {
///    eprintln!("Problem parsing argument: {}", err);
///    process::exit(1);
/// });
/// ```
pub fn run(config: Config) -> Result<(), Box<dyn Error>> {
    let contents = fs::read_to_string(config.filename)?;

    let result = if config.case_sensitive {
        search(&config.query, &contents)
    } else {
        search_insensitive(&config.query, &contents)
    };

    for line in result {
        println!("{}", &line)
    }

    Ok(())
}

// 重新导出
pub use self::grep::OsType;

pub mod grep {
    #[derive(Debug)]
    pub enum OsType {
        Windows,
        Linux,
        MaOS,
        FreeBSD,
    }

    impl OsType {
        fn new(t: OsType) -> OsType {
            OsType::FreeBSD
        }

        fn get(&self) -> OsType {
            OsType::MaOS
        }
    }
}

pub struct Config {
    pub query: String,
    pub filename: String,
    pub case_sensitive: bool,
}

impl Config {
    pub fn new(mut args: env::Args) -> Result<Config, String> {
        if args.len() < 3 {
            return Err("arguments not enough".to_string());
        }

        // 丢弃命令本身名称
        args.next();

        let query = match args.next() {
            None => {
                return Err("Can not get query args".into());
            }
            Some(v) => v,
        };

        let filename = match args.next() {
            None => {
                return Err("Can not get file name".into());
            }
            Some(v) => v,
        };

        let case_sensitive = env::var("CASE_INSENSITIVE").is_err();

        Ok(Config {
            query,
            filename,
            case_sensitive,
        })
    }
}

/// 执行搜索逻辑
///
/// # Examples
///
/// ```
/// let contents = fs::read_to_string(config.filename)?;
/// let result = if config.case_sensitive {
///     search(&config.query, &contents)
/// };
/// ```
pub fn search<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    contents
        .lines()
        .filter(|line| line.contains(query))
        .collect()
}

/// 执行搜索逻辑
///
/// # Examples
///
/// ```
/// let contents = fs::read_to_string(config.filename)?;
/// let result = if !config.case_sensitive {
///     search_insensitive(&config.query, &contents)
/// };
/// ```
pub fn search_insensitive<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
    let mut result = Vec::new();
    let query = query.to_lowercase();
    for line in contents.lines() {
        if line.to_lowercase().contains(&query) {
            result.push(line);
        }
    }
    result
}

#[cfg(test)]
mod tests {
    use std::vec;

    use super::*;

    #[test]
    fn one_result() {
        let query = "duct";
        let contents = "\
Rust:
safe, fast, productive.
Pick three.";
        assert_eq!(vec!["safe, fast, productive."], search(query, contents));
    }
}
