use std::{collections::HashMap, fmt::Display, fs::File, path::Path};

use bindgen::Builder;
use serde::Deserialize;

pub trait BindgenExt {
    fn config<P>(self, path: P) -> Result<Builder, Box<dyn std::error::Error>>
    where
        P: AsRef<Path> + Display;
}

impl BindgenExt for Builder {
    fn config<P>(mut self, path: P) -> Result<Builder, Box<dyn std::error::Error>>
    where
        P: AsRef<Path> + Display,
    {
        println!("cargo:rerun-if-changed={}", path.to_string());

        let cfg: BindgenCfg = serde_yaml::from_reader(File::open(path)?)?;

        for item in cfg.block {
            self = self.blocklist_item(item);
        }

        for func in cfg.function.allow {
            self = self.allowlist_function(func);
        }

        for func in cfg.function.block {
            self = self.blocklist_function(func);
        }

        for var in cfg.variable.allow {
            self = self.allowlist_var(var);
        }

        for typ in cfg.typ.allow {
            self = self.allowlist_type(typ);
        }

        for typ in cfg.typ.block {
            self = self.blocklist_type(typ);
        }

        for typ in cfg.opaque {
            self = self.opaque_type(typ);
        }

        for (enm, gen_method) in cfg.enm {
            self = match gen_method {
                BindgenEnum::Bitfield => self.bitfield_enum(enm),
                BindgenEnum::Constified => self.constified_enum(enm),
                BindgenEnum::ConstifiedModule => self.constified_enum_module(enm),
                BindgenEnum::NewType => self.newtype_enum(enm),
                BindgenEnum::Rustified => self.rustified_enum(enm),
                BindgenEnum::RustifiedNonExhaustive => self.rustified_non_exhaustive_enum(enm),
            };
        }

        Ok(self)
    }
}

#[derive(Debug, Deserialize, Default)]
pub struct BindgenCfg {
    #[serde(default, alias = "blacklist")]
    pub block: Vec<String>,
    #[serde(default)]
    pub function: BindgenFilter,
    #[serde(default)]
    pub variable: BindgenFilter,
    #[serde(rename = "type", default)]
    pub typ: BindgenFilter,
    #[serde(rename = "enum", default)]
    pub enm: HashMap<String, BindgenEnum>,
    #[serde(default)]
    pub opaque: Vec<String>,
}

#[derive(Debug, Deserialize, Default)]
pub struct BindgenFilter {
    #[serde(default, alias = "whitelist")]
    pub allow: Vec<String>,
    #[serde(default, alias = "blacklist")]
    pub block: Vec<String>,
}

#[derive(Debug, Deserialize)]
#[serde(rename_all = "snake_case")]
pub enum BindgenEnum {
    #[serde(alias = "b")]
    Bitfield,
    #[serde(alias = "nt")]
    NewType,
    #[serde(alias = "r")]
    Rustified,
    #[serde(alias = "rne")]
    RustifiedNonExhaustive,
    #[serde(alias = "c")]
    Constified,
    #[serde(alias = "cm")]
    ConstifiedModule,
}
