//! Enumerates all file types that **bindet** is able to detect.
//!
//! Those enumerations are separated in two:
//!
//! - [FileType]
//! - [FileRootType]

/// All file types that **bindet** is able to detect, some of them are just different versions
/// of the same format or very similar.
///
/// **bindet** is able to do both, refined detection, which it tries to detect [FileType] precisely,
/// and general detection, which it tries to do a weaker [FileType] detection
/// (**note**: in its current state, it is only doing the precise detection).
///
/// A more general categorization is enumerated by [FileRootType].
#[derive(Debug, Copy, Clone, Eq, PartialEq, Hash)]
#[non_exhaustive]
pub enum FileType {
    /// Zip File format, this includes:
    /// - jar
    /// - docx/xlsx/pptx
    /// - Apk
    /// - Aar
    /// - odt/ods/odp
    Zip,
    Rar,
    Rar5,
    /// Tar archive (only uncompressed).
    ///
    /// Compressed tar archives are first archived using tar, then
    /// compressed, which means that in order to detect compressed tar
    /// files we need to decompress them first, which is not viable
    /// for this library, and is not part of its goal.
    ///
    Tar,
    Lzma,
    Xz,
    Zst,
    Png,
    Jpg,
    _7z,
    /// Opus Ogg
    Opus,
    /// Vorbis Ogg
    Vorbis,
    Mp3,
    Webp,
    Flac,
    /// All Matroska media containers:
    /// - mkv
    /// - mka
    /// - mks
    /// - mk3d
    /// - webm
    Matroska,
    /// WebAssembly
    Wasm,
    /// Java class
    Class,
    /// Scala Tasty
    Tasty,
    /// Mach-O (untested)
    Mach,
    /// Executable and Linkable Format
    /// - `.so`
    /// - `*nix executable`
    Elf,
    /// Waveform Audio File Format
    Wav,
    Avi,
    /// Audio Interchange File Format
    /// - aiff
    /// - aif
    /// - aifc
    /// - snd
    /// - iff
    Aiff,
    /// Tagged Image File Format
    /// - Tiff
    /// - Tif
    Tiff,
    /// SQLite 3 Database
    Sqlite3,
    Ico,
    /// Dalvik (untested)
    Dalvik,
    /// PDF Document
    Pdf,
    /// DOS MZ Executable
    /// - `.exe`
    /// - `.dll`
    DosMzExecutable,
    /// DOS ZM Executable
    /// - `.exe`
    DosZmExecutable,
    /// Gimp XCF
    Xcf,
    /// Gif
    Gif,
    /// Bitmap
    Bmp,
    /// Optical Disc Image
    Iso,
}

impl FileType {
    /// Gets all variants of [FileType]
    pub fn variants() -> [FileType; 35] {
        [
            FileType::Zip,
            FileType::Rar,
            FileType::Rar5,
            FileType::Tar,
            FileType::Lzma,
            FileType::Xz,
            FileType::Zst,
            FileType::Png,
            FileType::Jpg,
            FileType::_7z,
            FileType::Opus,
            FileType::Vorbis,
            FileType::Mp3,
            FileType::Webp,
            FileType::Flac,
            FileType::Matroska,
            FileType::Wasm,
            FileType::Class,
            FileType::Tasty,
            FileType::Mach,
            FileType::Elf,
            FileType::Wav,
            FileType::Avi,
            FileType::Aiff,
            FileType::Tiff,
            FileType::Sqlite3,
            FileType::Ico,
            FileType::Dalvik,
            FileType::Pdf,
            FileType::DosMzExecutable,
            FileType::DosZmExecutable,
            FileType::Xcf,
            FileType::Gif,
            FileType::Bmp,
            FileType::Iso,
        ]
    }

    /// Gets the [FileRootType] of this [FileType]
    pub fn root(&self) -> FileRootType {
        match &self {
            FileType::Zip => FileRootType::Zip,
            FileType::Rar | FileType::Rar5 => FileRootType::Rar,
            FileType::Tar => FileRootType::Tar,
            FileType::Lzma => FileRootType::Lzma,
            FileType::Xz => FileRootType::Xz,
            FileType::Zst => FileRootType::Zst,
            FileType::Png => FileRootType::Png,
            FileType::Jpg => FileRootType::Jpg,
            FileType::_7z => FileRootType::_7z,
            FileType::Opus | FileType::Vorbis => FileRootType::Ogg,
            FileType::Mp3 => FileRootType::Mp3,
            FileType::Webp => FileRootType::Webp,
            FileType::Flac => FileRootType::Flac,
            FileType::Matroska => FileRootType::Matroska,
            FileType::Wasm => FileRootType::Wasm,
            FileType::Class => FileRootType::Class,
            FileType::Tasty => FileRootType::Tasty,
            FileType::Mach => FileRootType::Mach,
            FileType::Elf => FileRootType::Elf,
            FileType::Wav => FileRootType::Wav,
            FileType::Avi => FileRootType::Avi,
            FileType::Aiff => FileRootType::Aiff,
            FileType::Tiff => FileRootType::Tiff,
            FileType::Sqlite3 => FileRootType::Sqlite3,
            FileType::Ico => FileRootType::Ico,
            FileType::Dalvik => FileRootType::Dalvik,
            FileType::Pdf => FileRootType::Pdf,
            FileType::DosMzExecutable | FileType::DosZmExecutable => FileRootType::DosExecutable,
            FileType::Xcf => FileRootType::Xcf,
            FileType::Gif => FileRootType::Gif,
            FileType::Bmp => FileRootType::Bmp,
            FileType::Iso => FileRootType::Iso,
        }
    }
}

/// Root file type which categorizes types in a generic way.
///
/// For example, `Rar` and `Rar5` are both in the same `Rar` category, the same as
/// `DosMzExecutable` and `DosZmExecutable`, which are in the `DosExecutable` category.
///
/// This is useful when you are interested in knowing the type of a file, but does not need
/// much precision.
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
#[non_exhaustive]
pub enum FileRootType {
    Zip,
    Rar,
    Tar,
    Lzma,
    Xz, // XZ and LZMA in the same category?? I don't really know because LZMA is really old
    Zst,
    Png,
    Jpg,
    _7z,
    Ogg,
    Mp3,
    Webp,
    Flac,
    Matroska,
    Wasm,
    Class,
    Tasty,
    Mach,
    Elf,
    Wav,
    Avi,
    Aiff,
    Tiff,
    Sqlite3,
    Ico,
    Dalvik,
    Pdf,
    DosExecutable,
    Xcf,
    Gif,
    Bmp,
    Iso,
}

impl From<FileType> for FileRootType {
    fn from(file_type: FileType) -> Self {
        file_type.root()
    }
}
