//     bindet - Fast binary file type detection
//
//         The MIT License (MIT)
//
//      Copyright (c) Obliter Software (https://github.com/oblitersoftware/)
//      Copyright (c) contributors
//
//      Permission is hereby granted, free of charge, to any person obtaining a copy
//      of this software and associated documentation files (the "Software"), to deal
//      in the Software without restriction, including without limitation the rights
//      to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//      copies of the Software, and to permit persons to whom the Software is
//      furnished to do so, subject to the following conditions:
//
//      The above copyright notice and this permission notice shall be included in
//      all copies or substantial portions of the Software.
//
//      THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//      IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//      FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//      AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//      LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//      OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//      THE SOFTWARE.
//! File Type module

/// FileType
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum FileType {
    Zip,
    Rar,
    Rar5,
    /// Tar archive, regardless the compression
    Tar,
    Png,
    Jpg,
    _7z,
    /// Opus Ogg
    Opus,
    /// Vorbis Ogg
    Vorbis,
    Mp3,
    Webp,
    Flac,
    /// All Matroska media containers:
    /// - mkv
    /// - mka
    /// - mks
    /// - mk3d
    /// - webm
    Matroska,
    /// WebAssembly
    Wasm,
}

impl FileType {
    /// Gets all variants of [FileType]
    pub fn variants() -> [FileType; 14] {
        [
            FileType::Zip,
            FileType::Rar,
            FileType::Rar5,
            FileType::Tar,
            FileType::Png,
            FileType::Jpg,
            FileType::_7z,
            FileType::Opus,
            FileType::Vorbis,
            FileType::Mp3,
            FileType::Webp,
            FileType::Flac,
            FileType::Matroska,
            FileType::Wasm,
        ]
    }

    /// Gets the [FileRootType] of this [FileType]
    pub fn root(&self) -> FileRootType {
        match &self {
            FileType::Zip => FileRootType::Zip,
            FileType::Rar | FileType::Rar5 => FileRootType::Rar,
            FileType::Tar => FileRootType::Tar,
            FileType::Png => FileRootType::Png,
            FileType::Jpg => FileRootType::Jpg,
            FileType::_7z => FileRootType::_7z,
            FileType::Opus => FileRootType::Ogg,
            FileType::Vorbis => FileRootType::Ogg,
            FileType::Mp3 => FileRootType::Mp3,
            FileType::Webp => FileRootType::Webp,
            FileType::Flac => FileRootType::Flac,
            FileType::Matroska => FileRootType::Matroska,
            FileType::Wasm => FileRootType::Wasm,
        }
    }
}

///
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum FileRootType {
    Zip,
    Rar,
    Tar,
    Png,
    Jpg,
    _7z,
    Ogg,
    Mp3,
    Webp,
    Flac,
    Matroska,
    Wasm,
}
