//     bindet - Fast binary file type detection
//
//         The MIT License (MIT)
//
//      Copyright (c) Obliter Software (https://github.com/oblitersoftware/)
//      Copyright (c) contributors
//
//      Permission is hereby granted, free of charge, to any person obtaining a copy
//      of this software and associated documentation files (the "Software"), to deal
//      in the Software without restriction, including without limitation the rights
//      to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//      copies of the Software, and to permit persons to whom the Software is
//      furnished to do so, subject to the following conditions:
//
//      The above copyright notice and this permission notice shall be included in
//      all copies or substantial portions of the Software.
//
//      THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//      IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//      FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//      AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//      LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//      OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//      THE SOFTWARE.
//! Matcher module
use crate::types::FileType;

#[derive(Copy, Clone, Eq, PartialEq)]
pub enum RelativePosition {
    /// Relative to the position 0 of the file
    Start,
    /// Relative to the end of the file
    End,
}

#[derive(Copy, Clone, Eq, PartialEq)]
pub enum Step {
    /// Small buffer step
    Small,
    /// Large buffer step
    Large,
}

#[derive(Copy, Clone, Eq, PartialEq)]
pub enum TestResult {
    /// File type perfectly matches.
    Matched,
    /// File type partially matched, a second step may change the state to [TestResult::Matched].
    Maybe,
    /// File type not matched.
    NotMatched,
}

impl TestResult {
    fn or_else<F>(self, other: F) -> TestResult
    where
        F: FnOnce() -> TestResult,
    {
        return if self == TestResult::Matched {
            TestResult::Matched
        } else {
            let other = other();
            self | other
        };
    }

    fn and<F>(self, other: F) -> TestResult
    where
        F: FnOnce() -> TestResult,
    {
        return if self == TestResult::Matched || self == TestResult::Maybe {
            let other = other();
            self & other
        } else {
            TestResult::NotMatched
        };
    }
}

impl std::ops::BitOr for TestResult {
    type Output = TestResult;

    fn bitor(self, rhs: Self) -> Self::Output {
        return if self == TestResult::Matched || rhs == TestResult::Matched {
            TestResult::Matched
        } else if self == TestResult::Maybe && rhs == TestResult::Maybe {
            TestResult::Matched
        } else {
            TestResult::NotMatched
        };
    }
}

impl std::ops::BitAnd for TestResult {
    type Output = TestResult;

    fn bitand(self, rhs: Self) -> Self::Output {
        return if self == TestResult::Matched && rhs == TestResult::Matched {
            TestResult::Matched
        } else if self == TestResult::Maybe && rhs == TestResult::Maybe {
            TestResult::Maybe
        } else if self == TestResult::Matched && rhs == TestResult::Maybe {
            TestResult::Matched
        } else if self == TestResult::Maybe && rhs == TestResult::Matched {
            TestResult::Matched
        } else {
            TestResult::NotMatched
        };
    }
}

pub trait FileTypeMatcher {
    /// Tests if the [FileType] magic number can be found in the `bytes` slice.
    fn test(&self, relative_position: &RelativePosition, step: &Step, bytes: &[u8]) -> TestResult;
}

impl FileTypeMatcher for FileType {
    fn test(&self, relative_position: &RelativePosition, step: &Step, bytes: &[u8]) -> TestResult {
        match &self {
            FileType::Zip => zip_test(&relative_position, step, &bytes),
            FileType::Rar => rar_test(&relative_position, &bytes),
            FileType::Rar5 => rar5_test(&relative_position, &bytes),
            FileType::Tar => tar_test(&relative_position, step, &bytes),
            FileType::Png => png_test(&relative_position, step, &bytes),
            FileType::Jpg => jpg_test(&relative_position, step, &bytes),
            FileType::_7z => _7z_test(&relative_position, step, &bytes),
            FileType::Opus => opus_test(&relative_position, step, &bytes),
            FileType::Vorbis => vorbis_test(&relative_position, step, &bytes),
            FileType::Mp3 => mp3_test(&relative_position, step, &bytes),
            FileType::Webp => webp_test(&relative_position, step, &bytes),
            FileType::Flac => flac_test(&relative_position, step, &bytes),
            FileType::Matroska => matroska_test(&relative_position, step, &bytes),
            FileType::Wasm => wasm_test(&relative_position, step, &bytes),
        }
    }
}

/// Local File Header Signature
const LFHS: [u8; 4] = [0x50, 0x4b, 0x03, 0x04];
/// End of central directory signature
const ECDS: [u8; 4] = [0x50, 0x4b, 0x05, 0x06];

fn zip_test(relative_position: &RelativePosition, step: &Step, bytes: &[u8]) -> TestResult {
    return if *relative_position == RelativePosition::Start {
        if bytes.len() < 4 || *step != Step::Small {
            TestResult::NotMatched
        } else {
            let header = &bytes[..4];
            if header == LFHS {
                TestResult::Maybe
            } else {
                TestResult::NotMatched
            }
        }
    } else {
        if bytes.len() < 4 {
            TestResult::NotMatched
        } else {
            let signature = &bytes[..4];
            return if signature == ECDS {
                TestResult::Matched
            } else {
                TestResult::Maybe
            };
        }
    };
}

const RAR_4_MARKER: [u8; 7] = [0x52, 0x61, 0x72, 0x21, 0x1A, 0x07, 0x00];
fn rar_test(relative_position: &RelativePosition, bytes: &[u8]) -> TestResult {
    if *relative_position == RelativePosition::Start {
        return windowing_test(bytes, &RAR_4_MARKER);
    }

    return TestResult::NotMatched;
}

const RAR_5_MARKER: [u8; 8] = [0x52, 0x61, 0x72, 0x21, 0x1A, 0x07, 0x01, 0x00];
fn rar5_test(relative_position: &RelativePosition, bytes: &[u8]) -> TestResult {
    if *relative_position == RelativePosition::Start {
        return windowing_test(bytes, &RAR_5_MARKER);
    }

    return TestResult::NotMatched;
}

fn windowing_test(bytes: &[u8], matching: &[u8]) -> TestResult {
    if bytes.len() < matching.len() {
        return TestResult::NotMatched;
    }

    let mut i = 0;
    loop {
        if i < (bytes.len() - matching.len()) {
            let size = i + matching.len();
            let slice = &bytes[i..size];

            if slice == matching {
                return TestResult::Matched;
            } else {
                let mut any = false;
                for i in slice {
                    if *i == matching[0] {
                        any = true;
                    }
                }
                if !any {
                    i += (matching.len() - 1);
                }
            }

            i += 1
        } else {
            break;
        }
    }

    return TestResult::NotMatched;
}

const TAR_MARKER: [u8; 8] = [0x75, 0x73, 0x74, 0x61, 0x72, 0x00, 0x30, 0x30];
fn tar_test(relative_position: &RelativePosition, step: &Step, bytes: &[u8]) -> TestResult {
    if *relative_position == RelativePosition::Start && *step == Step::Small {
        if bytes.len() >= 257 + 8 {
            let slice = &bytes[257..257 + 8];
            if slice == TAR_MARKER {
                return TestResult::Matched;
            }
        }
    }
    TestResult::NotMatched
}

const PNG_SIGNATURE: [u8; 8] = [0x89, 0x50, 0x4e, 0x47, 0x0D, 0x0A, 0x1A, 0x0A];
fn png_test(relative_position: &RelativePosition, step: &Step, bytes: &[u8]) -> TestResult {
    if *relative_position == RelativePosition::Start && *step == Step::Small {
        return slice_test(&bytes, &PNG_SIGNATURE);
    }

    return TestResult::NotMatched;
}

const JPG_SIGNATURE: [u8; 2] = [0xFF, 0xD8];
fn jpg_test(relative_position: &RelativePosition, step: &Step, bytes: &[u8]) -> TestResult {
    if *relative_position == RelativePosition::Start && *step == Step::Small {
        return slice_test(&bytes, &JPG_SIGNATURE);
    }

    return TestResult::NotMatched;
}

const _7Z_SIGNATURE: [u8; 6] = [0x37, 0x7a, 0xBC, 0xAF, 0x27, 0x1C];
fn _7z_test(relative_position: &RelativePosition, step: &Step, bytes: &[u8]) -> TestResult {
    if *relative_position == RelativePosition::Start && *step == Step::Small {
        return slice_test(&bytes, &_7Z_SIGNATURE);
    }

    return TestResult::NotMatched;
}

const OPUS_START_SIGNATURE: [u8; 4] = [0x4f, 0x67, 0x67, 0x53];
const OPUS_INTERVAL: usize = 24;
const OPUS_HEAD_SIGNATURE: [u8; 8] = [0x4f, 0x70, 0x75, 0x73, 0x48, 0x65, 0x61, 0x64];
fn opus_test(relative_position: &RelativePosition, step: &Step, bytes: &[u8]) -> TestResult {
    if *relative_position == RelativePosition::Start && *step == Step::Small {
        if bytes.len() >= 36 {
            if let TestResult::Matched = slice_test(&bytes, &OPUS_START_SIGNATURE) {
                let start = OPUS_START_SIGNATURE.len() + OPUS_INTERVAL;
                return slice_test(&bytes[start..], &OPUS_HEAD_SIGNATURE);
            }
        }
    }

    return TestResult::NotMatched;
}

const VORBIS_INTERVAL: usize = 25;
const VORBIS_HEAD_SIGNATURE: [u8; 6] = [0x76, 0x6f, 0x72, 0x62, 0x69, 0x73];
fn vorbis_test(relative_position: &RelativePosition, step: &Step, bytes: &[u8]) -> TestResult {
    if *relative_position == RelativePosition::Start && *step == Step::Small {
        if bytes.len() >= 35 {
            if let TestResult::Matched = slice_test(&bytes, &OPUS_START_SIGNATURE) {
                let start = OPUS_START_SIGNATURE.len() + VORBIS_INTERVAL;
                return slice_test(&bytes[start..], &VORBIS_HEAD_SIGNATURE);
            }
        }
    }

    return TestResult::NotMatched;
}

const MP3_SIGNATURE_1: [u8; 2] = [0xFF, 0xFB];
const MP3_SIGNATURE_2: [u8; 2] = [0xFF, 0xF3];
const MP3_SIGNATURE_3: [u8; 2] = [0xFF, 0xF2];
const MP3_SIGNATURE_4: [u8; 3] = [0x49, 0x44, 0x43];
fn mp3_test(relative_position: &RelativePosition, step: &Step, bytes: &[u8]) -> TestResult {
    if *relative_position == RelativePosition::Start && *step == Step::Small {
        return slice_test(&bytes, &MP3_SIGNATURE_1)
            .or_else(|| slice_test(&bytes, &MP3_SIGNATURE_2))
            .or_else(|| slice_test(&bytes, &MP3_SIGNATURE_3))
            .or_else(|| slice_test(&bytes, &MP3_SIGNATURE_4));
    }

    return TestResult::NotMatched;
}

const WEBP_SIGNATURE_1: [u8; 4] = [0x52, 0x49, 0x46, 0x46];
const WEBP_SIGNATURE_2: [u8; 4] = [0x57, 0x45, 0x42, 0x50];
fn webp_test(relative_position: &RelativePosition, step: &Step, bytes: &[u8]) -> TestResult {
    if *relative_position == RelativePosition::Start && *step == Step::Small {
        if bytes.len() >= 12 {
            return slice_test(&bytes, &WEBP_SIGNATURE_1)
                .and(|| slice_test(&bytes[8..], &WEBP_SIGNATURE_2));
        }
    }

    return TestResult::NotMatched;
}

const FLAC_SIGNATURE: [u8; 4] = [0x66, 0x4C, 0x61, 0x43];
fn flac_test(relative_position: &RelativePosition, step: &Step, bytes: &[u8]) -> TestResult {
    if *relative_position == RelativePosition::Start && *step == Step::Small {
        return slice_test(&bytes, &FLAC_SIGNATURE);
    }

    return TestResult::NotMatched;
}

const MATROSKA_SIGNATURE: [u8; 4] = [0x1A, 0x45, 0xDF, 0xA3];
fn matroska_test(relative_position: &RelativePosition, step: &Step, bytes: &[u8]) -> TestResult {
    if *relative_position == RelativePosition::Start && *step == Step::Small {
        return slice_test(&bytes, &MATROSKA_SIGNATURE);
    }

    return TestResult::NotMatched;
}

const WASM_SIGNATURE: [u8; 4] = [0x00, 0x61, 0x73, 0x6D];
fn wasm_test(relative_position: &RelativePosition, step: &Step, bytes: &[u8]) -> TestResult {
    if *relative_position == RelativePosition::Start && *step == Step::Small {
        return slice_test(&bytes, &WASM_SIGNATURE);
    }

    return TestResult::NotMatched;
}

fn slice_test(bytes: &[u8], matching: &[u8]) -> TestResult {
    if bytes.len() >= matching.len() {
        if &bytes[..matching.len()] == matching {
            return TestResult::Matched;
        }
    }

    return TestResult::NotMatched;
}
