//! Create Sub-merchant API used for merchant/partner.

use serde::{Deserialize, Serialize};

#[derive(Debug, Serialize)]
#[serde(rename_all = "camelCase")]
pub struct Request {
    ///	The sub merchant name maximum length 128, unique under one mainMerchantId.
    pub merchant_name: String,

    /// 1=Personal(Individual)、2=solo proprietor、 3=Partnership、4=Private company、5=Others company
    pub merchant_type: u8,

    ///	Specified code	MCC Code, get from Binance
    pub merchant_mcc: String,

    ///	sub merchant logo url
    pub brand_logo: Option<String>,

    /// Iso alpha 2 country code(<https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2>),
    /// use "GO" if global	Country/Region of Business Operation,Can be multiple, split by "," eg:"SG,US"
    pub country: String,

    ///	store address
    pub address: Option<String>,

    ///	The legal name that is used in the registration, Required if merchantType is not Individual
    pub company_name: Option<String>,

    ///	Registration number/Company tax ID, Required if merchantType is not Individual
    pub registration_number: Option<String>,

    ///	Iso alpha 2 country code(<https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2>),
    ///	Country of Registration, Required if merchantType is not Individual
    pub registration_country: Option<String>,

    ///	Country of Registration, Required if merchantType is not Individual
    pub registration_address: Option<String>,

    ///	UnixTimestamp in milliseconds. The date when the business registration is in effective, Required if merchantType is not Individual
    pub incorporation_date: Option<u64>,

    ///	0=Online 1=Physical
    pub store_type: Option<u8>,

    ///	1,2,3,4	1=Web 2=App 3=Binance applets 4=Others , Required if merchantType is not Individual
    pub site_type: Option<u8>,

    ///	The URL of the website, Required if siteType is Web
    pub site_url: Option<String>,

    ///	The name of the website, Required if siteType is Web or App or Binance applets
    pub site_name: Option<String>,

    ///	1=ID 2=Passport, Required if merchantType is Individual
    pub certificate_type: Option<u8>,

    ///	iso alpha 2 country code(<https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2>),
    /// Required if merchantType is Individual
    pub certificate_country: Option<String>,

    ///	Required if merchantType is Individual
    pub certificate_number: Option<String>,

    ///	UnixTimestamp in milliseconds	Certificate Valid Date, Required if merchantType is Individual
    pub certificate_valid_date: Option<u64>,

    ///	UnixTimestamp in milliseconds	Contract date with ISV
    pub contract_time_isv: Option<u64>,
}

#[derive(Debug, Deserialize)]
#[cfg_attr(test, derive(Serialize))]
#[serde(rename_all = "camelCase")]
pub struct Response {
    /// Unique sub-merchant id generated by payment
    pub sub_merchant_id: u64,
}

#[cfg(test)]
mod tests {
    use crate::c2b::tests::test_request_serialize_deserialize;

    test_request_serialize_deserialize!(
        (
            test_create_sub_merchant_request_serialize,
            r#"
            {
                "merchantName": "Individual",
                "merchantType": 1,
                "merchantMcc": "5511",
                "brandLogo": null,
                "country": "CN,US",
                "address": null,
                "companyName": null,
                "registrationNumber": null,
                "registrationCountry": null,
                "registrationAddress": null,
                "incorporationDate": null,
                "storeType": null,
                "siteType": null,
                "siteUrl": null,
                "siteName": null,
                "certificateType": 1,
                "certificateCountry": "US",
                "certificateNumber": "123456X",
                "certificateValidDate": 1752422400000,
                "contractTimeIsv": 1594656000000
            }
            "#,
            Request {
                merchant_name: "Individual".to_string(),
                merchant_type: 1,
                merchant_mcc: "5511".to_string(),
                brand_logo: None,
                country: "CN,US".to_string(),
                address: None,
                company_name: None,
                registration_number: None,
                registration_country: None,
                registration_address: None,
                incorporation_date: None,
                store_type: None,
                site_type: None,
                site_url: None,
                site_name: None,
                certificate_type: Some(1),
                certificate_country: Some("US".to_string()),
                certificate_number: Some("123456X".to_string()),
                certificate_valid_date: Some(1752422400000),
                contract_time_isv: Some(1594656000000),
            }
        ),
        (
            test_create_sub_merchant_result_2_serialize,
            r#"
        {
            "merchantName": "Sole Proprietor",
            "merchantType": 2,
            "merchantMcc": "5511",
            "brandLogo": "logoUrlDemo",
            "country": "CN,US",
            "address": "store address demo",
            "companyName": "Sole Proprietor",
            "registrationNumber": "registration number demo",
            "registrationCountry": "US",
            "registrationAddress": "registration address demo",
            "incorporationDate": 1588262400000,
            "storeType": 1,
            "siteType": 2,
            "siteUrl": "site url demo",
            "siteName": "site name demo",
            "certificateType": null,
            "certificateCountry": null,
            "certificateNumber": null,
            "certificateValidDate": null,
            "contractTimeIsv": 1594656000000
        }
        "#,
            Request {
                merchant_name: "Sole Proprietor".to_string(),
                merchant_type: 2,
                merchant_mcc: "5511".to_string(),
                brand_logo: Some("logoUrlDemo".to_string()),
                country: "CN,US".to_string(),
                address: Some("store address demo".to_string()),
                company_name: Some("Sole Proprietor".to_string()),
                registration_number: Some("registration number demo".to_string()),
                registration_country: Some("US".to_string()),
                registration_address: Some("registration address demo".to_string()),
                incorporation_date: Some(1588262400000),
                store_type: Some(1),
                site_type: Some(2),
                site_url: Some("site url demo".to_string()),
                site_name: Some("site name demo".to_string()),
                certificate_type: None,
                certificate_country: None,
                certificate_number: None,
                certificate_valid_date: None,
                contract_time_isv: Some(1594656000000),
            }
        ),
        (
            test_create_sub_merchant_result_deserialize,
            r#"
            {    "subMerchantId": 2107268400000001  }
            "#,
            Response {
                sub_merchant_id: 2107268400000001,
            }
        )
    );
}
