//! Query order API used for merchant/partner to query order status

pub use crate::c2b::webhook::notification::order::TerminalType as TradeType;
use serde::{Deserialize, Serialize};

/// Either of the prepay id or the merchant trade no must be present.
#[derive(Serialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct Request {
    // maximum length 32,letter or digit, no other symbol allowed, can not be empty if prepayId is empty
    #[serde(skip_serializing_if = "Option::is_none")]
    prepay_id: Option<String>,

    // maximum length 19,letter or digit, no other symbol allowed, can not be empty if merchantTradeNo is empty
    #[serde(skip_serializing_if = "Option::is_none")]
    merchant_trade_no: Option<String>,
}

#[derive(Deserialize, Debug)]
#[cfg_attr(test, derive(Serialize))]
#[serde(rename_all = "UPPERCASE")]
pub enum Status {
    Initial,
    Pending,
    Paid,
    Canceled,
    Error,
    Refunding,
    Refunded,
    Expired,
}

#[derive(Deserialize, Debug)]
#[cfg_attr(test, derive(Serialize))]
#[serde(rename_all = "camelCase")]
pub struct Response {
    /// The merchant account id, issued when merchant been created at Binance.
    pub merchant_id: u64,

    /// unique id generated by binance
    pub prepay_id: String,

    /// Issued once the payment is successful
    pub transaction_id: Option<String>,

    /// merchant trade number
    pub merchant_trade_no: String,

    /// "WEB", "APP", "WAP", "MINI_PROGRAM", "PAYMENT_LINK", "OTHERS"	operate entrance
    pub trade_type: TradeType,

    /// "INITIAL", "PENDING", "PAID", "CANCELED", "ERROR", "REFUNDING", "REFUNDED", "EXPIRED"	order status
    pub status: Status,

    /// order currency
    pub currency: String,

    /// limitation refer to Create Order API order amount	order amount
    pub total_fee: f64,

    /// product name
    pub product_name: String,

    /// product detail
    pub product_detail: String,

    /// Consumer unique id
    pub open_user_id: String,

    /// UnixTimestamp in milliseconds when transaction happened
    pub transact_time: u64,

    /// UnixTimestamp in milliseconds when order was created
    pub create_time: u64,
}

impl Request {
    pub fn new(prepay_id: Option<String>, merchant_trade_no: Option<String>) -> Self {
        assert!(prepay_id.is_some() || merchant_trade_no.is_some());
        Self {
            prepay_id,
            merchant_trade_no,
        }
    }
}

#[cfg(test)]
mod tests {
    use crate::c2b::tests::test_request_serialize_deserialize;
    test_request_serialize_deserialize!(
        (
            test_query_order_serialization,
            r#"{"merchantTradeNo":"9825382937292"}"#,
            Request::new(None, Some("9825382937292".into()))
        ),
        (
            test_query_order_result_deserialization,
            r#"
            {
                "merchantId":98729382672,
                "prepayId": "383729303729303",
                "transactionId": "23729202729220282",
                "merchantTradeNo": "9825382937292",
                "tradeType": "APP",
                "status": "PAID",
                "currency": "EUR",
                "totalFee": 10.88,
                "productName": "Ice Cream",
                "productDetail": "Greentea ice cream cone",
                "openUserId": "",
                "transactTime":1425744000123,
                "createTime":1425744000000
              }
            "#,
            Response {
                merchant_id: 98729382672,
                prepay_id: "383729303729303".into(),
                transaction_id: Some("23729202729220282".to_string()),
                merchant_trade_no: "9825382937292".into(),
                trade_type: TradeType::App,
                status: Status::Paid,
                currency: "EUR".into(),
                total_fee: 10.88,
                product_name: "Ice Cream".into(),
                product_detail: "Greentea ice cream cone".into(),
                open_user_id: "".into(),
                transact_time: 1425744000123,
                create_time: 1425744000000,
            }
        )
    );
}
