//! Query order API used for merchant/partner to query order status

pub use super::webhook::notification::order::TerminalType as TradeType;
use serde::{Deserialize, Serialize};

/// Either of the prepay id or the merchant trade no must be present.
#[derive(Serialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct QueryOrder {
    // maximum length 32,letter or digit, no other symbol allowed, can not be empty if prepayId is empty
    #[serde(skip_serializing_if = "Option::is_none")]
    prepay_id: Option<String>,

    // maximum length 19,letter or digit, no other symbol allowed, can not be empty if merchantTradeNo is empty
    #[serde(skip_serializing_if = "Option::is_none")]
    merchant_trade_no: Option<String>,
}

#[derive(Deserialize, Debug)]
#[serde(rename_all = "UPPERCASE")]
pub enum Status {
    Initial,
    Pending,
    Paid,
    Canceled,
    Error,
    Refunding,
    Refunded,
    Expired,
}

#[derive(Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct QueryOrderResult {
    /// The merchant account id, issued when merchant been created at Binance.
    pub merchant_id: u128,

    /// unique id generated by binance
    pub prepay_id: String,

    /// Issued once the payment is successful
    pub transaction_id: Option<String>,

    /// merchant trade number
    pub merchant_trade_no: String,

    /// "WEB", "APP", "WAP", "MINI_PROGRAM", "PAYMENT_LINK", "OTHERS"	operate entrance
    pub trade_type: TradeType,

    /// "INITIAL", "PENDING", "PAID", "CANCELED", "ERROR", "REFUNDING", "REFUNDED", "EXPIRED"	order status
    pub status: Status,

    /// order currency
    pub currency: String,

    /// limitation refer to Create Order API order amount	order amount
    pub total_fee: f64,

    /// product name
    pub product_name: String,

    /// product detail
    pub product_detail: String,

    /// Consumer unique id
    pub open_user_id: String,

    /// UnixTimestamp in milliseconds when transaction happened
    pub transact_time: u128,

    /// UnixTimestamp in milliseconds when order was created
    pub create_time: u128,
}

impl QueryOrder {
    pub fn new(prepay_id: Option<String>, merchant_trade_no: Option<String>) -> Self {
        assert!(prepay_id.is_some() || merchant_trade_no.is_some());
        Self {
            prepay_id,
            merchant_trade_no,
        }
    }
}
#[cfg(test)]
mod tests {
    use serde_json::Value;

    use super::*;

    #[test]
    fn test_query_order_request_serialization() {
        let expected_request = r#"{"merchantTradeNo":"9825382937292"}"#;
        let qo = QueryOrder::new(None, Some("9825382937292".into()));
        assert_eq!(
            serde_json::to_value(&qo).unwrap(),
            serde_json::from_str::<Value>(expected_request).unwrap()
        );
    }
}
