use eiktyrner::{content, HttpClient};
use crate::{Param, PathAndQueryBuilder, Result};

use crate::Client;
use crate::models::{
     CreatedView,
     DateTime,
     Guid,
     IncomingPaymentRequestView,
     IncomingPaymentsView,
     OutgoingPaymentStatusView,
     OutgoingPaymentStatusesView,
     OutgoingPaymentView,
     PaymentMatchResultsView,
     PaymentMatchesView,
     SwishPaymentStatusView,
     UnhandledPaymentView,
     UnhandledPaymentsView,
    
};


impl Client {
    
    

/// Cancels a Swish payment requests and removes the payment request on the end users Swish app on their cell phone. 
/// - **id**: Invoice action public id
pub async fn payments_cancel_swish_payment_request(&self,id: String,
                         
) -> Result<()> {


    let path_and_query = PathAndQueryBuilder::default()
        .url(Param::value("v1"))
        .url(Param::value("payments"))
        .url(Param::value("swish"))
        .url(Param::value(id))
        .build();






    let request = self.new_request(
        http::method::Method::DELETE,
        path_and_query,
    )
    
    .body(content::Empty).expect("setting json body");
    
    
    self.client.send::<_, content::Empty>(request).await
    
}
    
    

/// Creates/Initiates a Swish payment request that will start a payment and initiate the payment request on the end users Swish app on their cell phone. Get the status of the payment by polling the GET endpoint. For more information see 
/// - **id**: Invoice action public id
/// - **phone**: Payer phone number
/// - **message**: Message to show on completed payment. Max 50 letters
pub async fn payments_create_swish_payment_request(&self,id: String,
                         phone: String,
                         message: Option<String>,
                         
) -> Result<CreatedView> {


    let path_and_query = PathAndQueryBuilder::default()
        .url(Param::value("v1"))
        .url(Param::value("payments"))
        .url(Param::value("swish"))
        .url(Param::value(id))
        .query("phone", Param::value(phone))
        .query("message", Param::opt(message))
        .build();






    let request = self.new_request(
        http::method::Method::POST,
        path_and_query,
    )
    
    .body(content::Empty).expect("setting json body");
    
    self.client.send::<_, content::Json<_>>(request).await
    
}
    
    

/// Create a standalone outgoing payment from your bankgiro. Please note that no bookkeeping is made for these payments. Alternative is to use self invoice or supplier invoice where bookkeeping is made. Status of the outgoing payment is made through a call to 'Get outgoing payment statuses', 'Get outgoing payment status for a payment' or view the Outgoingpayments report in the Portal.
pub async fn payments_create_an_outgoing_payment(&self,body: OutgoingPaymentView,
                         
) -> Result<CreatedView> {


    let path_and_query = PathAndQueryBuilder::default()
        .url(Param::value("v1"))
        .url(Param::value("payments"))
        .url(Param::value("outgoingpayments"))
        .build();






    let request = self.new_request(
        http::method::Method::POST,
        path_and_query,
    )
    .body(content::Json(body)).expect("setting json body");
    
    self.client.send::<_, content::Json<_>>(request).await
    
}
    
    

/// Cancel a standalone outgoing payment. This is a pure API feature. Status of the outgoing payment is made through a call to 'Get outgoing payment statuses'
/// - **id**: Outgoing payment public id
pub async fn payments_delete_an_outgoing_payment(&self,id: Guid,
                         
) -> Result<()> {


    let path_and_query = PathAndQueryBuilder::default()
        .url(Param::value("v1"))
        .url(Param::value("payments"))
        .url(Param::value("outgoingpayments"))
        .url(Param::value(id))
        .build();






    let request = self.new_request(
        http::method::Method::DELETE,
        path_and_query,
    )
    
    .body(content::Empty).expect("setting json body");
    
    
    self.client.send::<_, content::Empty>(request).await
    
}
    
    

/// Remove an unhandled payment from the system and book it if needed. This can be used if the unhandled payments are for an invoice that was not created or exists in Billecta
/// - **paymentpublicid**: Unhandled payment id
/// - **bookkeepingaccount**: Bookkeeping account used for balancing
/// - **transactiondate**: Transaction date for the deletion/repayment
pub async fn payments_delete_unhandled_payment(&self,paymentpublicid: Guid,
                         bookkeepingaccount: Option<String>,
                         transactiondate: Option<DateTime>,
                         
) -> Result<()> {


    let path_and_query = PathAndQueryBuilder::default()
        .url(Param::value("v1"))
        .url(Param::value("payments"))
        .url(Param::value("unhandledpayments"))
        .query("paymentpublicid", Param::value(paymentpublicid))
        .query("bookkeepingaccount", Param::opt(bookkeepingaccount))
        .query("transactiondate", Param::opt(transactiondate))
        .build();






    let request = self.new_request(
        http::method::Method::DELETE,
        path_and_query,
    )
    
    .body(content::Empty).expect("setting json body");
    
    
    self.client.send::<_, content::Empty>(request).await
    
}
    
    

/// Gets the status for the Swish payment request. A payment request must first be created/initiated by the POST endpoint. Use the returned token in the POST endpoint to retrieve status.
/// - **id**: Swish payment token
pub async fn payments_get_swish_payment_status(&self,id: String,
                         
) -> Result<SwishPaymentStatusView> {


    let path_and_query = PathAndQueryBuilder::default()
        .url(Param::value("v1"))
        .url(Param::value("payments"))
        .url(Param::value("swish"))
        .url(Param::value(id))
        .build();






    let request = self.new_request(
        http::method::Method::GET,
        path_and_query,
    )
    
    .body(content::Empty).expect("setting json body");
    
    self.client.send::<_, content::Json<_>>(request).await
    
}
    
    

/// Gets all incoming payments that has been matched/connected to an invoice. Unhandled/Unknown payments that are not related to any invoice will not be retrieved in this method. Please refer to 'Get unhandled payments' endpoint.
pub async fn payments_get_incoming_payments(&self,body: IncomingPaymentRequestView,
                         
) -> Result<IncomingPaymentsView> {


    let path_and_query = PathAndQueryBuilder::default()
        .url(Param::value("v1"))
        .url(Param::value("payments"))
        .url(Param::value("incomingpayments"))
        .build();






    let request = self.new_request(
        http::method::Method::PUT,
        path_and_query,
    )
    .body(content::Json(body)).expect("setting json body");
    
    self.client.send::<_, content::Json<_>>(request).await
    
}
    
    

/// Retrieves the status of single outgoing payment.
/// - **id**: Payment public id
pub async fn payments_get_outgoing_payment_status(&self,id: Guid,
                         
) -> Result<OutgoingPaymentStatusView> {


    let path_and_query = PathAndQueryBuilder::default()
        .url(Param::value("v1"))
        .url(Param::value("payments"))
        .url(Param::value("outgoingpayments"))
        .url(Param::value(id))
        .build();






    let request = self.new_request(
        http::method::Method::GET,
        path_and_query,
    )
    
    .body(content::Empty).expect("setting json body");
    
    self.client.send::<_, content::Json<_>>(request).await
    
}
    
    

/// Retrieves the status of all outgoing payments between two dates.
/// - **id**: Creditor public id
/// - **from**: Payments created from date
/// - **to**: Payments created to date
pub async fn payments_get_outgoing_payment_statuses_list_of_payments(&self,id: Guid,
                         from: DateTime,
                         to: DateTime,
                         
) -> Result<OutgoingPaymentStatusesView> {


    let path_and_query = PathAndQueryBuilder::default()
        .url(Param::value("v1"))
        .url(Param::value("payments"))
        .url(Param::value("outgoingpayments"))
        .url(Param::value(id))
        .query("from", Param::value(from))
        .query("to", Param::value(to))
        .build();






    let request = self.new_request(
        http::method::Method::GET,
        path_and_query,
    )
    
    .body(content::Empty).expect("setting json body");
    
    self.client.send::<_, content::Json<_>>(request).await
    
}
    
    

/// Retrieves all unhandled payments. An unhandled payment is an incoming payment that can't be connected to any invoice due to that the reference (OCR/invoice number/sender/etc) is unknown and can't be found in the system. Unhandled payments are handled through the MatchPayments endpoint.
/// - **id**: Creditor public id
pub async fn payments_get_unhandled_payments(&self,id: Guid,
                         
) -> Result<UnhandledPaymentsView> {


    let path_and_query = PathAndQueryBuilder::default()
        .url(Param::value("v1"))
        .url(Param::value("payments"))
        .url(Param::value("unhandledpayments"))
        .url(Param::value(id))
        .build();






    let request = self.new_request(
        http::method::Method::GET,
        path_and_query,
    )
    
    .body(content::Empty).expect("setting json body");
    
    self.client.send::<_, content::Json<_>>(request).await
    
}
    
    

/// Connects an unhandled payment that the system automatically couldn't connect.
/// - **id**: Creditor public id
pub async fn payments_match_unhandled_payments(&self,id: Guid,
                         body: PaymentMatchesView,
                         
) -> Result<PaymentMatchResultsView> {


    let path_and_query = PathAndQueryBuilder::default()
        .url(Param::value("v1"))
        .url(Param::value("payments"))
        .url(Param::value("matchpayments"))
        .url(Param::value(id))
        .build();






    let request = self.new_request(
        http::method::Method::POST,
        path_and_query,
    )
    .body(content::Json(body)).expect("setting json body");
    
    self.client.send::<_, content::Json<_>>(request).await
    
}
    
    

/// Creates an unhandled or over payment. This endpoint is only available in test enviroment
pub async fn payments_only_for_test_enviroment_create_unhadled_over_payment(&self,body: UnhandledPaymentView,
                         
) -> Result<CreatedView> {


    let path_and_query = PathAndQueryBuilder::default()
        .url(Param::value("v1"))
        .url(Param::value("payments"))
        .url(Param::value("unhandledpayments"))
        .build();






    let request = self.new_request(
        http::method::Method::POST,
        path_and_query,
    )
    .body(content::Json(body)).expect("setting json body");
    
    self.client.send::<_, content::Json<_>>(request).await
    
}
    
    

/// Sets the status of single outgoing payment. This endpoint is only available in test enviroment
/// - **id**: Payment public id
/// - **wassuccessful**: Succeded or failed
pub async fn payments_only_for_test_enviroment_mark_outgoing_payment_as_succeded_failed(&self,id: Guid,
                         wassuccessful: bool,
                         
) -> Result<()> {


    let path_and_query = PathAndQueryBuilder::default()
        .url(Param::value("v1"))
        .url(Param::value("payments"))
        .url(Param::value("outgoingpayments"))
        .url(Param::value(id))
        .query("wassuccessful", Param::value(wassuccessful))
        .build();






    let request = self.new_request(
        http::method::Method::PUT,
        path_and_query,
    )
    
    .body(content::Empty).expect("setting json body");
    
    
    self.client.send::<_, content::Empty>(request).await
    
}
    
}