use http::{
    self,
    method::Method,
    uri::{self, Uri},
};

use std::convert::TryFrom;

pub struct Client {
    auth_token: String,
    target: Uri,
    pub(crate) client: eiktyrner::HttpsClient,
}

impl Client {
    pub fn new<T, S: AsRef<str>>(target: T, token: S) -> Self
    where
        Uri: TryFrom<T>,
        <Uri as TryFrom<T>>::Error: std::fmt::Debug,
    {
        let target = Uri::try_from(target).expect("Invalid Billecta target");

        Self {
            client: eiktyrner::HttpsClient::default(),
            auth_token: format!("SecureToken {}", token.as_ref()),
            target,
        }
    }

    pub(crate) fn new_request(
        &self,
        method: Method,
        path_and_query: uri::PathAndQuery,
    ) -> http::request::Builder {
        let target = format!("{}{}", self.target, path_and_query.as_str())
            .parse::<Uri>()
            .map_err(|err| {
                format!(
                    "Failed to build target with {}: {}",
                    path_and_query.as_str(),
                    err
                )
            })
            .expect("Building new request");

        http::Request::builder()
            .method(method)
            .header("Authorization", &self.auth_token)
            .uri(target)
    }
}