use std::error::Error as StdError;
use std::fmt;

pub type ApiResult<T> = std::result::Result<T, crate::error::ApiError>;

#[derive(Debug)]
pub enum ApiError {
    Network(reqwest::Error),
    Remote(String),
}

impl ApiError {
    pub fn remote_err<T: ToString>(msg: T) -> Self {
        Self::Remote(msg.to_string())
    }

    pub fn is_network(&self) -> bool {
        matches!(self, Self::Network(_))
    }

    pub fn as_network(&self) -> Option<&reqwest::Error> {
        match self {
            Self::Network(e) => Some(&e),
            _ => None,
        }
    }
}

impl From<reqwest::Error> for ApiError {
    fn from(error: reqwest::Error) -> Self {
        Self::Network(error)
    }
}

impl fmt::Display for ApiError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> Result<(), fmt::Error> {
        match self {
            Self::Network(e) => e.fmt(f),
            Self::Remote(s) => write!(f, "Remote: {}", &s),
        }
    }
}

impl StdError for ApiError {
    fn source(&self) -> Option<&(dyn StdError + 'static)> {
        match self {
            Self::Network(e) => e.source(),
            _ => None,
        }
    }
}
