// module that does the conversion
use bfom_lib::{
    utilities::processing::{copy_css, file_find_markdown, uppercase_first_letter},
    Converter,
};

use std::{fs, io::Error, path::Path, path::PathBuf};

// to convert the toml into
use serde::Deserialize;

fn main() -> Result<(), Error> {
    println!("Stating BFOM converter.");
    let converter = Controller::new();
    converter.run()?;

    println!("Finished BFOM converter.");
    Ok(())
}

#[derive(Debug, Deserialize)]
pub struct Config {
    indentation: Option<usize>,
    src: Option<PathBuf>,
    dest: Option<PathBuf>,
    html_void: Option<Vec<String>>,
    template: Option<Template>,
}

#[derive(Debug, Deserialize)]
pub struct Template {
    enable: Option<bool>,
    general: Option<PathBuf>,
    order: Option<Vec<String>>,
}

#[derive(Debug, Clone)]
struct ConverterTemplate {
    enable: bool,
    general: Option<PathBuf>,
    order: Vec<String>,
}

#[derive(Debug)]
struct DefaultTemplateRow<'a> {
    data: &'a str,
    depth: usize,
}

#[derive(Debug)]
pub struct Controller {
    html_void: Vec<String>,
    indentation: usize,
    src: PathBuf,
    dest: PathBuf,
    template: ConverterTemplate,
}

impl Default for Controller {
    fn default() -> Self {
        Self::new()
    }
}

// uses templates to get the desired result
impl Controller {
    // takes the external config if provided and returns an instance of itself
    pub fn new() -> Self {
        // see if there is a config file
        let config_override = Controller::converter_get_config();

        // defaults set here
        let mut indentation: usize = 2;
        let mut html_void = vec!["area", "base", "br", "col", "command", "embed", "hr", "img", "input", "keygen", "link", "meta", "param", "source", "track", "wbr"]
            .iter()
            .map(|item| item.to_string())
            .collect::<Vec<String>>();

        let mut src: PathBuf = "./src".parse().unwrap();
        let mut dest: PathBuf = "./build".parse().unwrap();

        let mut template = ConverterTemplate { enable: false, general: None, order: vec![] };

        if let Some(config) = config_override {
            if let Some(value) = config.html_void {
                html_void = value;
            }
            if let Some(value) = config.indentation {
                indentation = value;
            }
            if let Some(value) = config.src {
                src = value;
            }
            if let Some(value) = config.dest {
                dest = value;
            }

            if let Some(template_file) = config.template {
                if let Some(value) = template_file.enable {
                    template.enable = value
                }
                // this is an option anyways
                template.general = template_file.general;

                if let Some(value) = template_file.order {
                    template.order = value
                }
            }
        }

        Controller { html_void, indentation, src, dest, template }
    }

    fn converter_get_config() -> Option<Config> {
        println!("Searching for .md.toml file");
        if let Ok(config_file) = fs::read_to_string(".md.toml") {
            if let Ok(config) = toml::from_str::<Config>(&*config_file) {
                println!("Found .md.toml, overriding defaults.");
                // the config file overrides teh defaults
                return Some(config);
            }
        }

        println!("No .md.toml file, using defaults.");
        None
    }

    // this manages the conversion
    pub fn run(&self) -> Result<(), Error> {
        // wipe the existing dest folder
        if self.dest.is_dir() {
            fs::remove_dir_all(&self.dest)?;
        }

        // get all teh files in teh specified folder
        let files = file_find_markdown(&self.src)?;

        copy_css(&self.src, &self.dest)?;

        //set up the converter
        let mut converter = Converter::new(self.html_void.clone(), self.indentation);

        // iterate through the files  replacing the extension, the src to dest
        for file in files {
            let src = file.clone();

            let mut dest = if let Ok(stripped) = file.clone().strip_prefix(&self.src) {
                Path::new(&self.dest).join(stripped)
            } else {
                // if the prefix cannot be stripped then put in the output files beside teh in
                // though considering that its based of of src this should never be called
                file.clone()
            };
            dest.set_extension("html");

            // we know the src folders exist, but check if the dest ones do
            if let Some(parent) = dest.parent() {
                fs::create_dir_all(parent)?
            }

            // get teh template, if applicable
            let (template, depth) = self.template_get(&src);

            // send it to the converter function
            let processed = converter.convert_file(&src, depth)?;

            // merge template and processed
            let merged = self.template_merge(&src, processed, template);

            // save it
            fs::write(&dest, merged)?;
        }

        Ok(())
    }

    fn template_get(&self, input: &Path) -> (Option<String>, usize) {
        if !self.template.enable {
            (None, 0)
        } else {
            let mut use_default = false;

            for item in &self.template.order {
                match item.as_str() {
                    "adjacent" => {
                        // check for <path>/<name>.html
                        let mut input_test = input.to_path_buf();
                        input_test.set_extension("html");

                        if let Ok(template_file) = fs::read_to_string(input_test) {
                            return (Some(template_file), 1);
                        }
                    }
                    "general" => {
                        // see if general is set in teh config
                        if let Some(general_path) = &self.template.general {
                            // then see if that files actually exists
                            if let Ok(template_file) = fs::read_to_string(general_path) {
                                return (Some(template_file), 1);
                            }
                        }
                    }
                    "folder" => {
                        // in the folder that the md file is in check if there is a .html file with the same name as the parent
                        let path = input.to_path_buf();
                        // yes this looks cursed to me too

                        if let Some(parent) = path.parent() {
                            if let Some(parent_name) = parent.file_name() {
                                let mut to_be_tested = PathBuf::from(parent_name);
                                to_be_tested.set_extension("html");
                                let to_test = parent.join(to_be_tested);
                                if let Ok(template_file) = fs::read_to_string(to_test) {
                                    return (Some(template_file), 1);
                                }
                            }
                        }
                    }
                    "default" => {
                        use_default = true;
                        break;
                    }
                    _ => {}
                }
            }

            if use_default {
                let default_data: Vec<DefaultTemplateRow> = vec![
                    DefaultTemplateRow { data: "<!DOCTYPE html>", depth: 0 },
                    DefaultTemplateRow { data: "<html lang='en'>", depth: 0 },
                    DefaultTemplateRow { data: "<head>", depth: 1 },
                    DefaultTemplateRow { data: "<title>{title}</title>", depth: 2 },
                    DefaultTemplateRow { data: "</head>", depth: 1 },
                    DefaultTemplateRow { data: "<body>", depth: 1 },
                    // handled by the processor itself
                    DefaultTemplateRow { data: "{body}", depth: 0 },
                    DefaultTemplateRow { data: "</body>", depth: 1 },
                    DefaultTemplateRow { data: "</html>", depth: 0 },
                ];

                let mut tmp = vec![];

                // add in teh appropriate indentation
                for row in default_data {
                    let indented = format!("{:indent$}{}", "", row.data, indent = (row.depth * self.indentation));
                    tmp.push(indented);
                }
                let template_default = tmp.join("\n");

                return (Some(template_default), 1);
            }

            // fallback to none
            (None, 0)
        }
    }

    fn template_merge(&self, input: &Path, processed: String, template: Option<String>) -> String {
        if let Some(template_string) = template {
            // use the file name for the title
            let mut input_test = input.to_path_buf();
            input_test.set_extension("");


            let title: String = if let Some(file_name) = input_test.file_name() {
                if let Some(result) = file_name.to_str() {
                    uppercase_first_letter(result)
                } else {
                    Default::default()
                }
            } else {
                Default::default()
            };

            // replace teh title and the body
            template_string.replace("{title}", &title).replace("{body}", &processed)
        } else {
            // no template, just return teh processed right back
            processed
        }
    }
}
