// module that does the conversion
use bfom_lib::{Config, Converter, Template};
use std::{fs, io::Error, path::PathBuf};

// to convert the toml into
use serde::Deserialize;

#[derive(Debug, Deserialize)]
struct MainConfig {
    indentation: Option<usize>,
    src: Option<PathBuf>,
    dest: Option<PathBuf>,
    html_void: Option<Vec<String>>,
    template: Option<MainTemplate>,
}
#[derive(Debug, Deserialize)]
pub struct MainTemplate {
    enable: Option<bool>,
    general: Option<PathBuf>,
    order: Option<Vec<String>>,
}

fn main() -> Result<(), Error> {
    println!("Stating BFOM converter.");
    let converter_config = converter_get_config();
    let mut converter = Converter::new(converter_config);
    converter.convert()?;

    println!("Finished BFOM converter.");
    Ok(())
}

fn converter_get_config() -> Option<Config> {
    println!("Searching for .md.toml file");
    if let Ok(config_file) = fs::read_to_string(".md.toml") {
        if let Ok(config) = toml::from_str::<MainConfig>(&*config_file) {
            println!("Found .md.toml, overriding defaults.");
            // the config file overrides teh defaults
            let converted = convert_config(config);
            return Some(converted);
        }
    }

    println!("No .md.toml file, using defaults.");
    None
}

fn convert_config(input: MainConfig) -> Config {
    let template = if let Some(template) = input.template {
        Some(Template {
            enable: template.enable,
            general: template.general,
            order: template.order,
        })
    } else {
        None
    };

    Config {
        indentation: input.indentation,
        src: input.src,
        dest: input.dest,
        html_void: input.html_void,
        template,
    }
}
