use std::collections::HashMap;
use std::fs::File;
use std::io::{BufRead, BufReader, Error};
use std::path::{Path, PathBuf};
use std::{fs, io};

// for toml stuff
use serde::Deserialize;

#[derive(Deserialize, Debug)]
pub struct Config {
    indentation: Option<usize>,
    src: Option<PathBuf>,
    dest: Option<PathBuf>,
    html_void: Option<Vec<String>>,
    template: Option<Template>,
}
#[derive(Deserialize, Debug)]
struct Template {
    enable: Option<bool>,
    general: Option<PathBuf>,
    order: Option<Vec<String>>,
}

// for converiubng
#[derive(Debug, Clone)]
struct NodeData {
    // h1/p/pre/etc
    // plain text is none
    tag: String,

    // any misc data, depends on teh tag
    misc: Option<String>,
    // the raw data inside this node
    text: Option<String>,

    contents: Vec<NodeData>,
}

#[derive(Debug, Clone)]
struct ReferenceLink {
    reference: String,
    url: String,
    title: Option<String>,
}

#[derive(Debug)]
pub struct Converter {
    html_void: Vec<String>,
    references: HashMap<String, ReferenceLink>,
    indentation: usize,
    src: PathBuf,
    dest: PathBuf,
    template: ConverterTemplate,
}

#[derive(Debug, Clone)]
struct ConverterTemplate {
    enable: bool,
    general: Option<PathBuf>,
    order: Vec<String>,
}

#[derive(Debug)]
struct DefaultTemplateRow<'a> {
    data: &'a str,
    depth: usize,
}

impl Converter {
    // takes the external config if provided and returns an isntance of itself
    pub fn new(config_override: Option<Config>) -> Converter {
        // see if there is a config file

        // defaults set here
        let mut indentation: usize = 2;
        let mut html_void = vec![
            "area", "base", "br", "col", "command", "embed", "hr", "img", "input", "keygen",
            "link", "meta", "param", "source", "track", "wbr",
        ]
        .iter()
        .map(|item| item.to_string())
        .collect::<Vec<String>>();

        let mut src: PathBuf = "./src".parse().unwrap();
        let mut dest: PathBuf = "./build".parse().unwrap();

        let mut template = ConverterTemplate {
            enable: false,
            general: None,
            order: vec![],
        };

        if let Some(config) = config_override {
            if let Some(value) = config.html_void {
                html_void = value;
            }
            if let Some(value) = config.indentation {
                indentation = value;
            }
            if let Some(value) = config.src {
                src = value;
            }
            if let Some(value) = config.dest {
                dest = value;
            }

            if let Some(template_file) = config.template {
                if let Some(value) = template_file.enable {
                    template.enable = value
                }
                // this is an option anyways
                template.general = template_file.general;

                if let Some(value) = template_file.order {
                    template.order = value
                }
            }
        }

        Converter {
            html_void,
            references: Default::default(),
            indentation,
            src,
            dest,
            template,
        }
    }

    // this manages the conversion
    pub fn convert(&mut self) -> Result<(), Error> {
        // wipe the existing dest folder
        fs::remove_dir_all(&self.dest)?;

        // get all teh files in teh specified folder
        let files = Converter::file_find_markdown(&self.src)?;

        self.copy_css(&self.src)?;

        // iterate through the files  replacing the extention, the src to dest
        for file in files {
            let src = file.clone();

            let mut dest = if let Ok(stripped) = file.clone().strip_prefix(&self.src) {
                Path::new(&self.dest).join(stripped)
            } else {
                // if the prefix cannot be stripped then put in the output files beside teh in
                // though considering that its based of of src this should never be called
                file.clone()
            };
            dest.set_extension("html");

            // we know the src folders exist, but check if the dest ones do
            if let Some(parent) = dest.parent() {
                fs::create_dir_all(parent)?
            }

            // send it to the converter function
            self.convert_file(&src, &dest)?;
        }

        Ok(())
    }

    pub fn convert_file(&mut self, input: &PathBuf, output: &PathBuf) -> Result<(), Error> {
        // read file into memory
        let lines = Converter::file_read(input)?;

        // grab the reference lines out of it
        let lines_processed_references = self.reference_link_get(lines);

        // get teh block elements
        let blocks = self.block_process(lines_processed_references, true)?;

        // get teh template, if applicable
        let (template, depth) = self.template_get(input);

        // merge it all together
        let processed = self.block_merge(blocks, depth);

        // merge template and processed
        let merged = self.template_merge(input, processed, template);

        // save
        Converter::file_write(merged, output)?;

        // clean up references
        self.references = Default::default();

        Ok(())
    }

    fn file_find_markdown(dir: &Path) -> io::Result<Vec<PathBuf>> {
        let mut markdown_files: Vec<PathBuf> = vec![];
        if dir.is_dir() {
            for entry in fs::read_dir(dir)? {
                let entry = entry?;
                let path = entry.path();
                if path.is_dir() {
                    let subfiles = Converter::file_find_markdown(&path)?;
                    markdown_files.extend(subfiles);
                } else {
                    // check if it ends with .md
                    if let Some(extension) = path.extension() {
                        if extension == "md" {
                            markdown_files.push(path)
                        }
                    }
                }
            }
        }
        Ok(markdown_files)
    }

    fn copy_css(&self, dir: &Path) -> io::Result<()> {
        if dir.is_dir() {
            for entry in fs::read_dir(dir)? {
                let entry = entry?;
                let path = entry.path();
                if path.is_dir() {
                    Converter::file_find_markdown(&path)?;
                } else {
                    // check if it ends with .css
                    if let Some(extension) = path.extension() {
                        if extension == "css" {
                            // grab the file

                            let dest = if let Ok(stripped) = path.clone().strip_prefix(&self.src) {
                                Path::new(&self.dest).join(stripped)
                            } else {
                                // if the prefix cannot be stripped then put in the output files beside teh in
                                // though considering that its based of of src this should never be called
                                path.clone()
                            };

                            // make sure it has a folder ot go into
                            if let Some(parent) = dest.parent() {
                                fs::create_dir_all(parent)?
                            }

                            fs::copy(&path, &dest)?;
                        }
                    }
                }
            }
        }
        Ok(())
    }

    // load the file into working memory
    fn file_read(path: &PathBuf) -> Result<Vec<String>, Error> {
        // read file into memory
        //println!("In file {}", path);
        // read the file into menory
        let input = File::open(path)?;
        let buffered = BufReader::new(input);

        let lines_all = buffered
            .lines()
            .map(|l| {
                let unwrapped = l.expect("Could not parse line");
                unwrapped.replace("\u{0000}", "\u{FFFD}")
            })
            .collect::<Vec<String>>();

        Ok(lines_all)
    }

    fn file_write(content: String, path: &PathBuf) -> Result<(), Error> {
        fs::write(path, content)?;
        Ok(())
    }

    fn template_get(&self, input: &PathBuf) -> (Option<String>, usize) {
        if !self.template.enable {
            (None, 0)
        } else {
            let mut use_default = false;

            for item in &self.template.order {
                match item.as_str() {
                    "adjacent" => {
                        // check for <path>/<name>.html
                        let mut input_test = input.clone();
                        input_test.set_extension("html");

                        if let Ok(template_file) = fs::read_to_string(input_test) {
                            return (Some(template_file), 1);
                        }
                    }
                    "general" => {
                        // see if genearal is set in teh config
                        if let Some(general_path) = &self.template.general {
                            // then see if that files actually exists
                            if let Ok(template_file) = fs::read_to_string(general_path) {
                                return (Some(template_file), 1);
                            }
                        }
                    }
                    "folder" => {
                        // in the folder that the md file is in check if there is a .html file with the same name as the parent
                        let path = input.clone();
                        // yes this looks cursed to me too

                        if let Some(parent) = path.parent() {
                            if let Some(parent_name) = parent.file_name() {
                                let mut to_be_tested = PathBuf::from(parent_name);
                                to_be_tested.set_extension("html");

                                if let Ok(template_file) = fs::read_to_string(to_be_tested) {
                                    return (Some(template_file), 1);
                                }
                            }
                        }
                    }
                    "default" => {
                        use_default = true;
                        break;
                    }
                    _ => {}
                }
            }

            if use_default {
                let default_data: Vec<DefaultTemplateRow> = vec![
                    DefaultTemplateRow {
                        data: "<!DOCTYPE html>",
                        depth: 0,
                    },
                    DefaultTemplateRow {
                        data: "<html lang='en'>",
                        depth: 0,
                    },
                    DefaultTemplateRow {
                        data: "<head>",
                        depth: 1,
                    },
                    DefaultTemplateRow {
                        data: "<title>{title}</title>",
                        depth: 2,
                    },
                    DefaultTemplateRow {
                        data: "</head>",
                        depth: 1,
                    },
                    DefaultTemplateRow {
                        data: "<body>",
                        depth: 1,
                    },
                    // handled by the processor itself
                    DefaultTemplateRow {
                        data: "{body}",
                        depth: 0,
                    },
                    DefaultTemplateRow {
                        data: "</body>",
                        depth: 1,
                    },
                    DefaultTemplateRow {
                        data: "</html>",
                        depth: 0,
                    },
                ];

                let mut tmp = vec![];

                // add in teh appropriate indentation
                for row in default_data {
                    let indented = format!(
                        "{:indent$}{}",
                        "",
                        row.data,
                        indent = (row.depth * self.indentation)
                    );
                    tmp.push(indented);
                }
                let template_default = tmp.join("\n");

                return (Some(template_default), 1);
            }

            // fallback to none
            (None, 0)
        }
    }

    fn template_merge(
        &self,
        input: &PathBuf,
        processed: String,
        template: Option<String>,
    ) -> String {
        if let Some(template_string) = template {
            // use the file name for the title
            let mut input_test = input.clone();
            input_test.set_extension("");

            let title = if let Some(file_name) = input_test.file_name() {
                if let Some(result) = file_name.to_str() {
                    result
                } else {
                    ""
                }
            } else {
                ""
            };

            // replace teh title and the body
            template_string
                .replace("{title}", title)
                .replace("{body}", &*processed)
        } else {
            // no template, just return teh processed right back
            processed
        }
    }

    fn reference_link_get(&mut self, lines: Vec<String>) -> Vec<String> {
        let mut result: Vec<String> = vec![];

        // iterate through the lines
        // if one matches the format it is not added to rteh reesult and added to teh link hash, if its not already there

        let mut references: HashMap<String, ReferenceLink> = HashMap::new();

        'outer: for line in lines {
            // only lines that start with [ are considered reference, number of spaces is irrelevent
            if !line.trim_start().starts_with('[') {
                result.push(line);
                continue;
            }

            // now check of the rest of the line matches

            // form manging position
            let characters: Vec<char> = line.trim_start().chars().collect();
            // starts at 1 to skip the first [
            let mut index_char = 1;

            // for getting teh identifiuer
            let mut reference_vec: Vec<char> = vec![];
            loop {
                if index_char >= characters.len() {
                    break;
                }

                let character = characters[index_char];

                match character {
                    ']' => {
                        // check if last character is a backslash
                        if characters[index_char - 1] == '\\' {
                            // remove teh slash
                            // add the bracket
                            reference_vec.pop();
                            reference_vec.push(character);
                        } else {
                            // incriment to account for this
                            index_char += 1;
                            break;
                        }
                    }
                    _ => reference_vec.push(character),
                }
                index_char += 1;
            }

            // need to havea n identifier
            if reference_vec.is_empty() {
                result.push(line);
                continue 'outer;
            }

            // check if next character is ':' (required)
            if index_char >= characters.len() {
                result.push(line);
                continue 'outer;
            }
            if characters[index_char] != ':' {
                // add to normal output
                result.push(line);
                continue 'outer;
            } else {
                // if ir is :
                index_char += 1;
            }

            // check if next character is whitespace, skip past tehse
            loop {
                if index_char >= characters.len() {
                    break;
                }
                let character = characters[index_char];

                if character == ' ' || character == '\t' {
                    index_char += 1;
                } else {
                    break;
                }
            }

            // now get teh url
            let mut url_vec: Vec<char> = vec![];
            // (start, end)
            let mut angle_brackets = (false, false);
            loop {
                if index_char >= characters.len() {
                    break;
                }
                let character = characters[index_char];

                match character {
                    '<' => {
                        // opening is only alloweed on teh first chr
                        if !angle_brackets.0 {
                            angle_brackets = (true, false);
                        } else {
                            url_vec.push(character)
                        }
                    }
                    '>' => {
                        // check if it was opened
                        if angle_brackets.0 {
                            // then this closes it out

                            // check if its presseded by backslash, its action is ignored
                            if characters[index_char - 1] == '\\' {
                                // remove teh slash
                                url_vec.pop();
                                url_vec.push(character);
                            } else {
                                // actually closes it out
                                angle_brackets = (true, true);

                                index_char += 1;
                                break;
                            }
                        } else {
                            url_vec.push(character);
                        }
                    }
                    ' ' | '\t' => {
                        // space breaks the url, unless its in an angled bracket

                        if angle_brackets.0 {
                            url_vec.push(character)
                        } else {
                            break;
                        }
                    }

                    _ => url_vec.push(character),
                }

                index_char += 1;
            }

            if url_vec.is_empty() {
                // check if its angle brackets
                if !angle_brackets.1 {
                    result.push(line);
                    continue 'outer;
                }
            }

            // if angle brackets were opened but never closed
            if angle_brackets.0 && !angle_brackets.1 {
                result.push(line);
                continue 'outer;
            }

            // check for whitespace again, but this time it must be at least 1 if its followed by an open quotation
            let mut whitespace = 0;
            loop {
                if index_char >= characters.len() {
                    break;
                }
                let character = characters[index_char];

                match character {
                    ' ' | '\t' => {
                        index_char += 1;
                        whitespace += 1;
                    }
                    '\'' | '"' => {
                        if whitespace == 0 {
                            result.push(line);
                            continue 'outer;
                        } else {
                            break;
                        }
                    }
                    _ => break,
                }
            }

            // find the optional title next

            // first char must be eitehr ' or "

            let mut title_quote: Option<char> = None;
            if index_char < characters.len() {
                let character = characters[index_char];
                match character {
                    '\'' | '"' => title_quote = Some(character),
                    _ => {}
                }
                index_char += 1;
            }

            let mut title_vec: Vec<char> = vec![];

            if let Some(delimiter) = title_quote {
                // starts off being opened
                let mut title_closed = false;

                loop {
                    if index_char >= characters.len() {
                        break;
                    }
                    let character = characters[index_char];

                    if character == delimiter {
                        // check if preceding was backslash

                        if characters[index_char - 1] == '\\' {
                            // remove teh slash
                            // add the bracket
                            title_vec.pop();
                            title_vec.push(character);
                        } else {
                            title_closed = true;

                            // increment to account for this
                            //index_char += 1;
                            break;
                        }
                    } else {
                        title_vec.push(character)
                    }

                    index_char += 1;
                }

                // check if it was closed properly
                if !title_closed {
                    // title is optional so just clear it instead of revoking teh line
                    title_vec = vec![];
                }
            }

            // tidy up
            let reference: String = reference_vec.iter().collect();
            let url: String = url_vec.iter().collect();

            let title: Option<String> = if title_vec.is_empty() {
                None
            } else {
                Some(title_vec.iter().collect())
            };

            let reference_data = ReferenceLink {
                reference: reference.clone(),
                url,
                title,
            };

            references.insert(reference, reference_data);
        }

        self.references = references;

        result
    }

    fn block_process(&self, lines: Vec<String>, _top_layer: bool) -> Result<Vec<NodeData>, Error> {
        // this gets most of the structure done

        // what gets returned
        let mut node_data: Vec<NodeData> = vec![];

        // these handle dealing with teh current lines
        let mut row: isize = 0;

        // These keep track of info about the current block
        let mut block_type: Option<String> = None;
        let mut block_start: usize = 0;
        let mut block_lines: Vec<String> = vec![];
        let mut block_excluded: Vec<String> = vec![];

        // specific for code blocks
        let mut pre_language: Option<String> = None;
        let mut pre_closer: String = Default::default();

        // soecificallyt for OL and ul lists
        let mut lists_lines: Vec<NodeData> = vec![];
        let mut lists_indent = 0;
        let mut lists_number: Option<String> = None;

        // for html tags, need to use this to find the closing tag
        let mut html_closing: Option<Vec<char>> = None;
        let mut html_tag = "".to_string();
        // sets the position a html tag ended on, if its nto teh whole line
        // (row, col)
        let mut html_end_col = (0, 0);
        let mut html_start = vec![];
        let mut html_depth = 0;

        let mut table_header_length = 0;
        let mut table_header: Vec<NodeData> = vec![];
        let mut table_rows: Vec<NodeData> = vec![];
        let mut table_alignment: HashMap<i32, String> = HashMap::new();

        'main: loop {
            // ensure that row is in bounds
            if row >= lines.len() as isize {
                break;
            }

            let line = if html_end_col.0 == row && html_end_col.1 != 0 {
                &lines[row as usize][html_end_col.1..]
            } else {
                lines[row as usize].as_str()
            };

            // check what type of line it is
            let line_current_type = Converter::block_process_line_type(line, &block_excluded);
            // check if the current line is teh last line
            let line_current_is_end = (row + 1) >= (lines.len() as isize);

            //println!("Start:  Top: {}. Row {} of {}. Block: {:?}, Current Line: {}. Line: {}. End?: {}", _top_layer, row, lines.len(), block_type, line_current_type.0, line, line_current_is_end);

            let mut block_finished = false;
            let mut block_reset = false;

            // block specific stuff

            let mut finished_p = false;
            let mut finished_pre = false;
            let mut finished_blockquote = false;

            let mut finished_xl = false;
            let mut finished_li = false;

            let mut finished_html = false;

            let mut finished_table = false;

            // now process based on what type of line it is
            match &block_type {
                // this handles first lines of everything
                None => {
                    // first line of the block, make sure the tmp array is cleared
                    block_lines = vec![];

                    // set the block start as well
                    block_start = row as usize;

                    // h and hr are inline and reset line_type after
                    match &line_current_type.0.as_str() {
                        &"h" => {
                            let mut count = 0;
                            let mut broken = false;
                            let mut content_array: Vec<char> = vec![];
                            for character in line.trim().chars() {
                                match character {
                                    // skip this one
                                    '#' => {
                                        if !broken && count < 6 {
                                            count += 1
                                        } else {
                                            content_array.push(character);
                                        }
                                    }
                                    // break on a space or closing tag <br/>
                                    _ => {
                                        broken = true;
                                        content_array.push(character);
                                    }
                                }
                            }

                            // for this the tag is depending on teh numebr of #'s at the beginning
                            let tag = format!("h{}", &count);

                            // clean si the raw with teh tag elements removed
                            let content_string = content_array.to_vec().iter().collect::<String>();
                            let text = Some(content_string.as_str().trim().to_string());

                            // add the data
                            node_data.push(NodeData {
                                tag,
                                text,
                                misc: None,
                                contents: vec![],
                            });

                            // reset it here
                            block_type = None;

                            // go straight to the next row
                            row += 1;
                            continue;
                        }
                        &"hr" => {
                            node_data.push(NodeData {
                                tag: "hr".to_string(),
                                text: None,
                                misc: None,
                                contents: vec![],
                            });
                            block_type = None;

                            // go straight to the next row
                            row += 1;
                            continue;
                        }

                        // blocks from here on out
                        // paragraph is the kinda catch all
                        &"p" => {
                            // paragraph is the catch all here
                            // anything that isnt caught in opther blocks ends up ehre

                            // first line of a paragraph

                            // check if tis whitespace, if it is then skip this line
                            // will still be None next time round
                            if line == "" {
                                row += 1;
                                continue;
                            }

                            let line_to_push = if block_excluded.contains(&"html".to_string())
                                || block_excluded.contains(&"html_comment".to_string())
                                || block_excluded.contains(&"html_cdata".to_string())
                            {
                                // strip first < from it, &gt;
                                line.replacen("<", "&gt;", 1).to_string()
                            } else {
                                line.to_string()
                            };

                            // add it to the tmp array
                            block_lines.push(line_to_push);

                            block_type = Some(line_current_type.0);

                            // no need for block_start as this is to catch teh dregs
                        }
                        &"pre" => {
                            // first line of pre
                            // only setting up the block,

                            // get the language
                            let mut pre_language_tmp: Vec<char> = vec![];
                            let mut closer: Vec<char> = vec![];
                            let mut space = false;
                            for character in line.trim().chars() {
                                match character {
                                    // skip this one
                                    '`' => {
                                        if !space {
                                            closer.push(character)
                                        }

                                        continue;
                                    }
                                    ' ' | '\t' => {
                                        // first one mark it as a space
                                        if !space {
                                            space = true;
                                        } else {
                                            pre_language_tmp.push(character);
                                        }
                                    }
                                    _ => {
                                        pre_language_tmp.push(character);
                                    }
                                }
                            }
                            pre_language = if pre_language_tmp.is_empty() {
                                None
                            } else {
                                let language = pre_language_tmp
                                    .iter()
                                    .collect::<String>()
                                    .trim()
                                    .to_string();

                                pre_closer = closer.iter().collect::<String>().trim().to_string();
                                if language == "" {
                                    None
                                } else {
                                    Some(language)
                                }
                            };

                            // set the block type
                            block_type = Some(line_current_type.0);
                        }

                        &"blockquote" => {
                            // like the p but every line is prefixed by >
                            // sort out the sub blocks when it is finished

                            // remove the first >
                            // add it to the tmp array
                            let cleaned = line.replacen(">", "", 1).replacen(" ", "", 1);
                            block_lines.push(cleaned);

                            // set the block type
                            block_type = Some(line_current_type.0);
                        }

                        &"ul*" | &"ul-" | &"ul+" => {
                            // * and -
                            // strip
                            let trimmed = line.trim_start();
                            let cleaned = if trimmed.starts_with("* ") {
                                trimmed.replacen("* ", "", 1)
                            } else {
                                trimmed.replacen("- ", "", 1)
                            };
                            block_lines.push(cleaned);

                            lists_indent = line_current_type.1.unwrap();

                            // set the block type
                            block_type = Some(line_current_type.0);
                        }

                        &"ol" => {
                            let vec_chars: Vec<char> = line.chars().collect();

                            // remove lists_indent's worth ofd characters
                            // let cleaned
                            let content_start = line_current_type.1.unwrap();
                            let trimmed = &vec_chars[content_start..];
                            let cleaned = trimmed.iter().collect::<String>();

                            block_lines.push(cleaned);

                            lists_indent = line_current_type.1.unwrap();

                            // only used for numbered lists
                            lists_number = Some(line_current_type.2.unwrap().to_string());

                            // set the block type
                            block_type = Some(line_current_type.0);
                        }

                        // html_php/html_comment/html_cdata
                        &"html" => {
                            /*

                            // commonmarks html stuff is basically a warcrime
                            // for example this breaks what you would expect https://spec.commonmark.org/0.30/#example-148
                            // best way I can figure doing this is to not have markdown in html, all html gets passed through.
                            // yes I know this is opinionated but it evades the issue in the example above
                            // Adding HTML to a markdown document should override the markdown, not teh other way round
                            // Some safety nets are removed like verifying urls are valid



                            // clean - tag starts on a new line and nothing after the closing tag
                            // one html block
                            // <tag>
                            // </tag>

                            // follwwed by html - a new html tag starts after this one,
                            // counts as two seperate html blocks
                            // <tag>
                            // </tag><tag2>
                            // </tag2>

                            // inline 1 - anything that would be classified as a paragraph after the tag
                            // all part of same paragraph block - html not touched by paragraph processing
                            // <tag></tag> same line

                            // inline 2 - anything that would be classified as a paragraph after the tag
                            // all part of same paragraph block - html not touched by paragraph processing
                            // <tag>
                            // </tag> same line as closing tag

                            // a line starting with < can be either be a html tag or an escaped url
                            // <tag></tag>
                            // <tag />
                            // not going to check if the url is valid, thats up to teh browser
                            // <protocol:url>
                            // <user@url>

                            */

                            // type of tag isnt known here

                            // this fuirst line is to find the tag name
                            let mut tag_vec: Vec<char> = vec![];

                            // 1 to skip the opening < in teh first run
                            let mut index = 1;
                            let line_chars: Vec<char> = line.chars().collect();
                            loop {
                                if index >= line_chars.len() {
                                    break;
                                }

                                let character = line_chars[index];

                                // set it to be used next round
                                index += 1;
                                match character {
                                    'a'..='z' | 'A'..='Z' | '0'..='9' => {
                                        tag_vec.push(character);
                                    }
                                    ':' => {
                                        // checkl to see if its a url
                                        if tag_vec.is_empty() {
                                            // if it is then go straight back to reprocessing it
                                            block_excluded.push(line_current_type.0);
                                            continue 'main;
                                        } else {
                                            break;
                                        }
                                    }
                                    _ => {
                                        // not a valid tag character
                                        break;
                                    }
                                }
                            }

                            // not a valid tag
                            if tag_vec.is_empty() {
                                // repeat teh current line, but not as a html tag
                                block_excluded.push(line_current_type.0);
                                continue;
                            }

                            // use html_void to see what type of closing it has
                            // if its in html_void then its />
                            // else its </tag>

                            let tag: String = tag_vec.iter().collect();

                            // check for nestled html
                            // <div>1<div>2</div></div>
                            //html_depth += 1;
                            html_start.push('<');
                            html_start.extend(&tag_vec);

                            // these are the self-closing tags
                            if self.html_void.contains(&tag) {
                                html_closing = Some(vec!['/', '>']);
                            } else {
                                let mut tmp = vec!['<', '/'];
                                tmp.extend(&tag_vec);
                                tmp.push('>');
                                html_closing = Some(tmp);
                            }

                            // set the type for the next round
                            // html_php/html_comment/html_cdata are handled by html in teh second round,
                            // only needed to be broken out in teh first round to deal with setting teh different end condition
                            block_type = Some(line_current_type.0);

                            // set the tag to be used lated for banned/permitted tags
                            html_tag = tag;
                        }

                        // these are much teh same as html, byt can skip directly to knowing teh closing tag
                        &"html_comment" => {
                            // -->
                            html_start = vec!['<', '!', '-', '-'];
                            html_closing = Some(vec!['-', '-', '>']);
                            block_type = Some(line_current_type.0);
                            html_tag = "custom_comment".to_string();
                        }

                        &"html_cdata" => {
                            // ]]>
                            html_start = vec!['<', '!', '[', 'C', 'D', 'A', 'T', 'A', '['];
                            html_closing = Some(vec![']', ']', '>']);
                            block_type = Some(line_current_type.0);
                            html_tag = "custom_cdata".to_string();
                        }

                        &"table" => {
                            /*
                            Mostly taken from https://github.github.com/gfm/#tables-extension- with a few changes:
                              First character on a line is always |, while a trailing | isnt explicity required it cna look better.
                              First like is always the header.
                                As a result no requirement for a delimiter line under it.
                                If you want multiple rows of headers, or spans please use a html table.
                              Delimiter lines are lines starting with |, with 1+ - 's, trailing | isnt required (like above) but does look better.
                                They are ignored in teh html, but make the table look better in markdown
                                All equal:
                                  |-
                                  |----------------|
                                  |--|---|----|----|
                              Table ends on teh first line not starting with |
                            */

                            // delimiter lines are ignored fo the table content
                            // but they do provide alignment
                            if line
                                .replace("|", "")
                                .replace("-", "")
                                .replace(":", "")
                                .trim()
                                == ""
                            {
                                // will use https://developer.mozilla.org/en-US/docs/Web/CSS/text-align as the way github does it is no longer valid

                                // alignment will impact subsequent lines, unless overwritten
                                table_alignment = Converter::block_process_table_alignment(line);

                                row += 1;
                                continue 'main;
                            }

                            let mut header_contents: Vec<NodeData> = vec![];

                            /*

                            from:
                            |th1|th2|..|thn

                            to:
                            NodeData {
                                tag: "th".to_string(),
                                misc: None,
                                text: Some("th1".to_string()),
                                contents: vec![
                                ]
                            },
                            */

                            // figure out how many headers there are

                            let mut content: Vec<char> = vec![];
                            let mut last_character: Option<char> = None;
                            for character in line.trim_end_matches('|').chars() {
                                match character {
                                    '|' => {
                                        if let Some(x) = last_character {
                                            if x == '\\' {
                                                // remove the backslash
                                                content.pop();
                                                content.push(character);
                                                last_character = Some(character);
                                                continue;
                                            }
                                        }

                                        if table_header_length > 0 {
                                            let text: String = content.iter().collect();
                                            let alignment = if let Some(x) =
                                                table_alignment.get(&(table_header_length as i32))
                                            {
                                                Some(x.clone())
                                            } else {
                                                None
                                            };

                                            // add current
                                            header_contents.push(NodeData {
                                                tag: "th".to_string(),
                                                misc: alignment,
                                                text: Some(text),
                                                contents: vec![],
                                            });
                                        }

                                        // add anotehr col
                                        table_header_length += 1;

                                        // clear teh existing data (since its previously used
                                        content = vec![];
                                    }
                                    _ => content.push(character),
                                }
                                last_character = Some(character);
                            }

                            // tidy up any trailing content
                            if !content.is_empty() {
                                let text: String = content.iter().collect();
                                let alignment = if let Some(x) =
                                    table_alignment.get(&(table_header_length as i32))
                                {
                                    Some(x.clone())
                                } else {
                                    None
                                };

                                // add current
                                header_contents.push(NodeData {
                                    tag: "th".to_string(),
                                    misc: alignment,
                                    text: Some(text),
                                    contents: vec![],
                                });
                            }

                            table_header.push(NodeData {
                                tag: "tr".to_string(),
                                misc: None,
                                text: None,
                                contents: header_contents,
                            });

                            block_type = Some(line_current_type.0);
                        }

                        _ => {
                            // just add it to teh array and deal with it when ti is finished
                            block_lines.push(line.to_string());

                            // mark it as a _ for teh next iteration
                            block_type = Some(line_current_type.0);
                        }
                    }
                }

                // this is second time+ round
                Some(line_type_previous) => {
                    let previous_line = line_type_previous.as_str();
                    match previous_line {
                        "p" => {
                            if &line_current_type.0 != "p" {
                                // if current line is not part of a paragraph
                                // dont add it to teh array
                                // go back one row to re-analyse it next run

                                // this is cancelled out later
                                row -= 1;

                                finished_p = true;
                            } else if line == "" {
                                // "" marks an empty line which is automatically the end of a paragraph
                                finished_p = true;
                            } else {
                                // normal action
                                block_lines.push(line.to_string());
                            };
                        }
                        "pre" => {
                            // trim line and check if it is the same as the opener
                            if line.trim() == pre_closer.as_str() {
                                // end of block, now tidy up
                                finished_pre = true;
                            } else {
                                // add the lines to teh block
                                block_lines.push(line.to_string());

                                // if its the last line then its caught alter down
                            }
                        }
                        "blockquote" => {
                            // like the paragraph there is no need to worry about reset as each line starts with >

                            if &line_current_type.0 != "blockquote" {
                                // dont add it to teh array
                                // go back one row to re-analyse it next run

                                // this is cancelled out later
                                row -= 1;

                                finished_blockquote = true;
                            } else {
                                let cleaned = line.replacen(">", "", 1).replacen(" ", "", 1);
                                block_lines.push(cleaned);
                            };
                        }
                        "ul*" | "ul-" | "ul+" | "ol" => {
                            // gotta do two things
                            // tody up teh current list entry block
                            // close out teh lsit

                            let indent_casted = lists_indent;
                            let leading_spaces =
                                String::from_utf8(vec![b' '; indent_casted]).unwrap();

                            if line.starts_with(&leading_spaces) {
                                // if it starts with lists_indent spaces its staying in teh same li
                                let trimmed = line.replacen(" ", "", indent_casted);
                                block_lines.push(trimmed);
                            } else if line == "" {
                                // if its "" its the same li
                                block_lines.push(line.to_string());
                            } else if line_current_type.0 == previous_line {
                                // if its ul then close up teh current li
                                // contents are not added to teh li
                                finished_li = true;

                                // repeat the current line next time
                                row -= 1;
                            } else {
                                // line is not part of the ul or li
                                // close out teh li and ul
                                finished_li = true;
                                finished_xl = true;

                                // reevaluate the line
                                row -= 1;
                            }
                        }

                        // html blocks
                        "html" | "html_comment" | "html_cdata" => {
                            // do nothing here, its handled below
                        }
                        "table" => {
                            if &line_current_type.0 != "table" {
                                // dont add it to teh array
                                // go back one row to re-analyse it next run

                                // this is cancelled out later
                                row -= 1;

                                finished_table = true;
                            } else {
                                // if its a delimiter then update the alignment, then skip to teh next row
                                if line
                                    .replace("|", "")
                                    .replace("-", "")
                                    .replace(":", "")
                                    .trim()
                                    == ""
                                {
                                    table_alignment =
                                        Converter::block_process_table_alignment(line);
                                    row += 1;
                                    continue 'main;
                                }

                                // process the current row
                                // can only have table_header_length cols, any less than that are filled in

                                let mut row_contents: Vec<NodeData> = vec![];

                                /*

                                from:
                                |cell1|cell2|..|celln

                                to:
                                NodeData {
                                    tag: "td".to_string(),
                                    misc: None,
                                    text: Some("th1".to_string()),
                                    contents: vec![
                                    ]
                                },
                                */

                                // figure out how many headers there are

                                let mut local_cols = 0;
                                let mut content: Vec<char> = vec![];
                                let mut last_character: Option<char> = None;
                                for character in line.chars() {
                                    match character {
                                        '|' => {
                                            // is this new col too many?
                                            if local_cols > table_header_length {
                                                content = vec![];
                                                break;
                                            }

                                            if let Some(x) = last_character {
                                                if x == '\\' {
                                                    // remove the backslash
                                                    content.pop();
                                                    content.push(character);
                                                    last_character = Some(character);
                                                    continue;
                                                }
                                            }

                                            // add previous data
                                            if local_cols > 0 {
                                                let text: String = content.iter().collect();
                                                let alignment = if let Some(x) =
                                                    table_alignment.get(&(local_cols as i32))
                                                {
                                                    Some(x.clone())
                                                } else {
                                                    None
                                                };

                                                // add current
                                                row_contents.push(NodeData {
                                                    tag: "td".to_string(),
                                                    misc: alignment,
                                                    text: Some(text),
                                                    contents: vec![],
                                                });
                                            }

                                            // add anotehr col
                                            local_cols += 1;

                                            // clear teh existing data (since its previously used
                                            content = vec![];
                                        }
                                        _ => content.push(character),
                                    }
                                }

                                // tidy up any trailing content
                                if !content.is_empty() {
                                    let text: String = content.iter().collect();
                                    let alignment = if let Some(x) =
                                        table_alignment.get(&(local_cols as i32))
                                    {
                                        Some(x.clone())
                                    } else {
                                        None
                                    };

                                    // add current
                                    row_contents.push(NodeData {
                                        tag: "td".to_string(),
                                        misc: alignment,
                                        text: Some(text),
                                        contents: vec![],
                                    });
                                }

                                // add in empty cells if its not up to teh header
                                while row_contents.len() < table_header_length {
                                    row_contents.push(NodeData {
                                        tag: "td".to_string(),
                                        misc: None,
                                        text: Some("".to_string()),
                                        contents: vec![],
                                    });
                                }

                                table_rows.push(NodeData {
                                    tag: "tr".to_string(),
                                    misc: None,
                                    text: None,
                                    contents: row_contents,
                                });
                            }
                        }

                        _ => {}
                    }
                }
            }

            // for specifically processing html blocks, fires off if there is a closing tag to match against
            if let Some(closing) = &html_closing {
                let mut index = 0;
                let line_chars: Vec<char> = line.chars().collect();

                let mut valid_chars_closing = 0;
                let mut valid_chars_opening = 0;
                loop {
                    if closing.is_empty() {
                        break;
                    }
                    if html_start.is_empty() {
                        break;
                    }
                    if index >= line_chars.len() {
                        break;
                    }

                    let character = line_chars[index];

                    // set it to be used next round
                    index += 1;

                    if html_start[valid_chars_opening] == character {
                        // character is valid,
                        valid_chars_opening += 1;

                        if valid_chars_opening == html_start.len() {
                            html_depth += 1;
                            valid_chars_opening = 0;
                        }
                    } else {
                        // reset
                        valid_chars_opening = 0;
                    }

                    if closing[valid_chars_closing] == character {
                        // character is valid,
                        valid_chars_closing += 1;

                        if valid_chars_closing == closing.len() {
                            html_depth -= 1;
                            if html_depth == 0 {
                                finished_html = true;
                                break;
                            }
                            // reset it
                            valid_chars_closing = 0;
                        }
                    } else {
                        // reset
                        valid_chars_closing = 0;
                    }
                }
                if finished_html {
                    let mut partial_line = false;
                    let mut permitted_in_p = false;
                    let forbidden_tag = false; // how handling forbidden tags would be

                    // check if there is anything left on teh line
                    if line[index..].trim() != "" {
                        partial_line = true;

                        // check if the tag is permitted in paragraphs
                        // https://developer.mozilla.org/en-US/docs/Web/Guide/HTML/Content_categories#phrasing_content
                        // https://www.w3.org/TR/html52/dom.html#phrasing-content
                        let allowed_tags_p = vec![
                            // pure html stuff
                            "abbr",
                            "audio",
                            "b",
                            "button",
                            "canvas",
                            "cite",
                            "code",
                            "data",
                            "datalist",
                            "dfn",
                            "em",
                            "embed",
                            "i",
                            "iframe",
                            "img",
                            "input",
                            "label",
                            "mark",
                            "math",
                            "meter",
                            "noscript",
                            "object",
                            "output",
                            "picture",
                            "progress",
                            "q",
                            "ruby",
                            "samp",
                            "script",
                            "select",
                            "small",
                            "span",
                            "string",
                            "sub",
                            "sup",
                            "svg",
                            "textarea",
                            "time",
                            "u",
                            "var",
                            "video",
                            "wbr",
                            // according to dn these are not always applicable
                            "a",
                            "del",
                            "ins",
                            "map",
                            // these require an itemprop attribite, however not looking for these
                            // "link", "mata",

                            // these are the "custom" tags for misc heml stuff
                            "custom_comment",
                            "custom_cdata",
                        ];

                        if allowed_tags_p.contains(&html_tag.as_str()) {
                            permitted_in_p = true;
                        }
                    }

                    // add list of forbidden tags here

                    if forbidden_tag {
                        // if its forbidden then its not apssed through
                        // mark teh block as finished
                        block_finished = true;

                        // but dont do teh tidy up of the html tag
                        finished_html = false;
                    }

                    // tidy up the closing of html here
                    if partial_line {
                        if permitted_in_p {
                            // change the type to paragraph
                            block_type = Some("p".to_string());

                            // add whole line to teh array
                            block_lines.push(line.to_string());

                            // need to mark this as false as its going to be taken over as a paragraph enxxt round
                            finished_html = false;

                            // reset teh end col
                            html_end_col = (0, 0);
                        } else {
                            // teh first part is always added to teh tmp array
                            let tmp_line = &line[..index];
                            block_lines.push(tmp_line.to_string());

                            // decide what to do with teh rest

                            // get the type of the ramining line
                            // if its html or paragraph then its processed
                            // else it is completely skipped
                            let remaining = &line[index..];
                            let (line_next, _, _) =
                                Converter::block_process_line_type(remaining, &block_excluded);

                            match line_next.as_str() {
                                "p" | "html" | "html_comment" | "html_cdata" => {
                                    // other stuff after the html tag, mark the end here
                                    html_end_col = (row, index);

                                    // repeat this row to see the new stuff
                                    row -= 1;
                                }
                                _ => {
                                    // make sure the end col is reset back as its not html or a paragraph
                                    html_end_col = (0, 0);
                                }
                            }
                        }
                    } else {
                        // make sure teh end col is reset back
                        html_end_col = (0, 0);

                        // add whole line to teh array
                        block_lines.push(line.to_string());
                    }
                } else {
                    // add it like normal
                    block_lines.push(line.to_string());
                }
            }

            // to catch end of line stuff
            // specifically one liners
            if line_current_is_end {
                match &block_type {
                    None => {}
                    Some(line_type_previous) => match line_type_previous.as_str() {
                        "p" => {
                            finished_p = true;
                        }
                        "blockquote" => {
                            finished_blockquote = true;
                        }
                        "ul*" | "ul-" | "ul+" | "ol" => {
                            finished_li = true;
                            finished_xl = true;
                        }
                        "pre" => {
                            if !finished_pre {
                                block_reset = true;
                            }
                        }
                        "html" | "html_comment" | "html_cdata" => {
                            if !finished_html {
                                block_reset = true;
                            }
                        }
                        "table" => {
                            finished_table = true;
                        }
                        _ => {}
                    },
                }
            }

            // close out blocks

            if finished_p {
                node_data.push(NodeData {
                    tag: "p".to_string(),
                    text: Some(block_lines.join("\n")),
                    misc: None,
                    contents: vec![],
                });

                block_finished = true;
            }

            // not impacted by line_current_is_end, needs to have its closing tag
            if finished_pre {
                // replace every < with &lt;
                let text = block_lines.join("\n").replace("<", "&lt;").to_string();
                node_data.push(NodeData {
                    tag: "pre".to_string(),
                    text: Some(text),
                    misc: pre_language.to_owned(),
                    contents: vec![],
                });

                block_finished = true;
            }

            if finished_blockquote {
                let contents = self
                    .block_process(block_lines.clone(), false)
                    // if it errors then fail fairly gracefully
                    .unwrap_or_default();

                node_data.push(NodeData {
                    tag: "blockquote".to_string(),
                    text: None,
                    misc: None,
                    contents,
                });

                block_finished = true;
            }

            if finished_li {
                // block line is always >= 1
                if block_lines.len() == 1 {
                    // the li has text in it
                    lists_lines.push(NodeData {
                        tag: "li".to_string(),
                        text: Some(block_lines[0].to_owned()),
                        misc: lists_number.clone(),
                        contents: vec![],
                    });
                } else {
                    let contents = self
                        .block_process(block_lines.clone(), false)
                        // if it errors then fail fairly gracefully
                        .unwrap_or_default();

                    lists_lines.push(NodeData {
                        tag: "li".to_string(),
                        text: None,
                        misc: lists_number.clone(),
                        contents,
                    });
                }

                // only used temporary
                block_lines = vec![];
                block_type = None;
                //lists_number = None;
            }

            if finished_xl {
                // if a number is there then its a order list
                let list_type = match lists_number {
                    None => "ul".to_string(),
                    Some(_) => "ol".to_string(),
                };

                node_data.push(NodeData {
                    tag: list_type,
                    text: None,
                    misc: None,
                    contents: lists_lines.clone(),
                });

                // cleanup
                block_finished = true;
            }

            if finished_html {
                node_data.push(NodeData {
                    tag: "html".to_string(),
                    // gonna do no more processing on this, just going to pass it straight through
                    text: Some(block_lines.join("\n")),
                    misc: None,
                    contents: vec![],
                });
                // cleanup
                block_finished = true;
            }

            if finished_table {
                let table = NodeData {
                    tag: "table".to_string(),
                    misc: None,
                    text: None,
                    contents: vec![
                        NodeData {
                            tag: "thead".to_string(),
                            misc: None,
                            text: None,
                            contents: table_header.clone(),
                        },
                        NodeData {
                            tag: "tbody".to_string(),
                            misc: None,
                            text: None,
                            contents: table_rows.clone(),
                        },
                    ],
                };

                node_data.push(table);

                block_finished = true;
            }

            // handle resets first
            if block_reset {
                // reset row to teh start
                row = block_start as isize;

                // make sure it dosent come down this path again
                block_excluded.push(block_type.unwrap_or_default().to_owned());

                // reset blocktype
                block_type = None;

                pre_language = None;
                pre_closer = Default::default();

                // reset when not in use
                lists_lines = vec![];
                lists_indent = 0;
                lists_number = None;

                // html stuff
                html_closing = None;
                html_tag = "".to_string();
                html_start = vec![];
                html_depth = 0;

                // tables
                table_header = vec![];
                table_rows = vec![];
                table_header_length = 0;
                table_alignment = HashMap::new();

                // go back to start, do not pass go
                continue;
            }

            // do the cleanup for a finished block
            if block_finished {
                // if the block was anything but html then set the end col back to 0
                if let Some(x) = &block_type {
                    if !x.contains("html") {
                        html_end_col = (0, 0);
                    }
                }

                // reset the type
                block_type = None;
                block_lines = vec![];

                // clear teh exclusions
                block_excluded = vec![];

                // reset the language for pre blocks
                pre_language = None;
                pre_closer = Default::default();

                // reset when not in use
                lists_lines = vec![];
                lists_indent = 0;
                lists_number = None;

                // html stuff
                html_closing = None;
                html_tag = "".to_string();
                html_start = vec![];
                html_depth = 0;

                // tables
                table_header = vec![];
                table_rows = vec![];
                table_header_length = 0;
                table_alignment = HashMap::new();
            }

            // incriment
            // this may be returning to the same row
            row += 1;
        }
        Ok(node_data)
    }

    fn block_process_table_alignment(line: &str) -> HashMap<i32, String> {
        let mut alignment: HashMap<i32, String> = HashMap::new();

        let mut last_character: Option<char> = None;
        let mut start = false;
        let mut end = false;
        let mut counter = 0;
        for character in line.trim_end_matches('|').chars() {
            match character {
                '|' => {
                    // skip first character
                    if last_character.is_none() {
                        // dont forget to set it here
                        last_character = Some(character);
                        continue;
                    }

                    counter += 1;

                    if start && end {
                        alignment.insert(counter, "center".to_string());
                    } else if start {
                        alignment.insert(counter, "left".to_string());
                    } else if end {
                        alignment.insert(counter, "right".to_string());
                    } else {
                        // dont insert
                    }

                    // reset for next one
                    start = false;
                    end = false;
                }
                ':' => {
                    // if preceded by a space or bar then its a start

                    // if preceded by - then its a end

                    if let Some(last) = last_character {
                        if last == ' ' || last == '|' {
                            start = true;
                        }
                        if last == '-' {
                            end = true;
                        }
                    }
                }
                _ => {
                    // nothing should happen here
                }
            }

            last_character = Some(character);
        }

        counter += 1;
        // tidy up the trailing
        if start && end {
            alignment.insert(counter, "center".to_string());
        } else if start {
            alignment.insert(counter, "left".to_string());
        } else if end {
            alignment.insert(counter, "right".to_string());
        } else {
            // dont insert
        }

        alignment
    }

    // (type, indent, number)
    fn block_process_line_type(
        line: &str,
        excluded: &[String],
    ) -> (String, Option<usize>, Option<i32>) {
        let trimmed = line.trim();
        let paragraph = ("p".to_string(), None, None);

        if !excluded.contains(&"h".to_string()) && trimmed.starts_with('#') {
            ("h".to_string(), None, None)
        }
        // remove dashes and spaces
        // if the result is empty then its a hr
        // if not it will probally become a paragraph
        else if !excluded.contains(&"hr".to_string())
            && trimmed.starts_with("----")
            && line.replace("-", "").replace(" ", "").replace("\t", "") == ""
        {
            ("hr".to_string(), None, None)
        } else if line.starts_with('>') {
            // spoiler
            if line.starts_with(">!") {
                return paragraph;
            }

            if !excluded.contains(&"blockquote".to_string()) {
                ("blockquote".to_string(), None, None)
            } else {
                paragraph
            }
        } else if !excluded.contains(&"pre".to_string()) && line.starts_with("```") {
            ("pre".to_string(), None, None)
        } else if trimmed.starts_with('<') {
            // html sorting here

            // "html"| "html_comment"| "html_cdata"

            // <!--
            if trimmed.starts_with("<!--") && !excluded.contains(&"html_comment".to_string()) {
                return ("html_comment".to_string(), None, None);
            }

            // <![CDATA[
            if trimmed.starts_with("<![CDATA[") && !excluded.contains(&"html_cdata".to_string()) {
                return ("html_cdata".to_string(), None, None);
            }

            // autolinks now start with <<
            if trimmed.starts_with("<<") {
                return paragraph;
            }

            if !excluded.contains(&"html".to_string()) {
                ("html".to_string(), None, None)
            } else {
                // paragraph is the catch all if all teh above fail
                paragraph
            }
        } else if !excluded.contains(&"table".to_string()) && trimmed.starts_with('|') {
            ("table".to_string(), None, None)
        }
        // trim coupled with searching for a space will clear out any blank list start rows
        else if !excluded.contains(&"ul*".to_string()) && trimmed.starts_with("* ") {
            // figure out indent,
            // starts at 2 to account for teh */-
            let mut indent = 2;
            for character in line.chars() {
                match character {
                    ' ' => {
                        indent += 1;
                    }
                    _ => break,
                }
            }

            ("ul*".to_string(), Some(indent), None)
        } else if !excluded.contains(&"ul-".to_string()) && trimmed.starts_with("- ") {
            // figure out indent,
            // starts at 2 to account for teh */-
            let mut indent = 2;
            for character in line.chars() {
                match character {
                    ' ' => {
                        indent += 1;
                    }
                    _ => break,
                }
            }

            ("ul-".to_string(), Some(indent), None)
        } else if !excluded.contains(&"ul+".to_string()) && trimmed.starts_with("+ ") {
            // figure out indent,
            // starts at 2 to account for teh */-
            let mut indent = 2;
            for character in line.chars() {
                match character {
                    ' ' => {
                        indent += 1;
                    }
                    _ => break,
                }
            }

            ("ul+".to_string(), Some(indent), None)
        }
        // this covers everything that isnt as easy to find as the above
        else {
            // /((^ {0,8})([0-9]{1,9})(\.{1})( {1})(\w|\d))/gm

            // first check teh first 12 characters for an ordered List
            // diverging from commonmark in this regard to align up to the limit:
            //         1. First Line
            // 123456789. Last Line
            //
            // I can do it as I am not recognising four spaces as a code block

            // up to 8 initial spaces
            // number
            // dot
            // space
            // Non space character

            let characters: Vec<char> = line.chars().collect();
            let mut index = 0;

            // determine initial spaces
            loop {
                if index >= characters.len() {
                    break;
                }

                let character = characters[index];

                match character {
                    ' ' => {
                        // do nothing
                    }
                    _ => break,
                }
                index += 1;
            }

            /* unlimited indentation allowed
            if index > 8 {
                // not valid
                // too many spaces
                return paragraph;
            }
             */

            // get numbers next
            let mut numbers: Vec<char> = vec![];
            loop {
                if index >= characters.len() {
                    break;
                }

                let character = characters[index];
                match character {
                    '0'..='9' => {
                        numbers.push(character);
                    }
                    _ => break,
                }
                index += 1;
            }

            if numbers.is_empty() {
                // not valid
                // needs a number
                return paragraph;
            }
            if numbers.len() > 9 {
                // not valid
                // too many numbers
                return paragraph;
            }

            // need one dot
            let mut dot = false;
            if index < characters.len() {
                let character = characters[index];

                if character == '.' {
                    dot = true;
                }

                if character == ')' {
                    dot = true;
                }

                index += 1;
            }

            if !dot {
                // not valid
                // needs a dot
                return paragraph;
            }

            let mut space = false;
            if index < characters.len() {
                let character = characters[index];

                if character == '.' {
                    space = true;
                }

                index += 1;
            }

            if !space {
                // not valid
                // needs a space
                return paragraph;
            }

            // all the above are valid to make it here

            let number_str: String = numbers.into_iter().collect();
            let number: i32 = number_str.parse().unwrap_or(0);

            ("ol".to_string(), Some(index), Some(number))
        }
    }

    fn block_merge(&self, nodes: Vec<NodeData>, depth: usize) -> String {
        let mut result: Vec<String> = vec![];

        for node in nodes {
            let tag = node.tag.as_str();
            let (opening, content, closing) = match tag {
                // html just gets passed straight through
                "html" => {
                    if let Some(text) = node.text {
                        result.push(text);
                    }
                    (None, None, None)
                }
                // simple stuff first, no recursion

                "h1" | "h2" | "h3" | "h4" | "h5" | "h6" => {
                    if let Some(text) = node.text {
                        let processed = self.inline_process(text, 0);

                        // no attributes or anything
                        let formatted = format!("<{}>{}</{}>", tag, processed, tag);

                        (Some(formatted), None, None)
                    }else{
                        (None, None, None)
                    }
                }

                "p" => {
                    if let Some(text) = node.text {
                        let processed = self.inline_process(text, depth+1);

                        // no attributes or anything
                        let open = format!("<{}>", tag);
                        let close = format!("</{}>", tag);

                        (Some(open), Some(processed), Some(close))
                    }else{
                        (None, None, None)
                    }
                }

                "hr" => {
                    // simplest of them all
                    (Some("<hr />".to_string()), None, None)
                }

                "pre" => {
                    if let Some(contents) = node.text {

                        // misc for fenced code blocks is used for the language
                        let formatted = if let Some(language) = node.misc {
                            // using class="language-{language}" is a psudo standard
                            format!("<pre><code class=\"language-{}\">\n{}\n</code></pre>", language, contents)
                        } else {
                            format!("<pre><code>\n{}\n</code></pre>", contents)
                        };

                        result.push(formatted);
                    }
                    (None, None, None)
                }

                ///////////////////////////
                // recursive beyond this //
                ///////////////////////////

                // table
                "table" | "thead" | "tbody"| "tr" |
                // blockquote always contains other stuff
                "blockquote" |
                // lists
                "ul" | "ol"
                => {
                    let processed = self.block_merge(node.contents, depth+1);
                    let open = format!("<{}>", tag);
                    let close = format!("</{}>", tag);
                    (Some(open), Some(processed), Some(close))
                }

                "th" | "td" => {
                    let processed = if let Some(text) = node.text {
                        self.inline_process(text, depth+1)
                    } else{
                        "".to_string()
                    };

                    let open = if let Some(alignment) = node.misc {
                        // text-align: alignment;
                        // https://developer.mozilla.org/en-US/docs/Web/CSS/text-align
                        format!("<{} style=\"text-align: {};\">", tag, alignment)
                    } else {
                        format!("<{}>", tag)
                    };

                    let close = format!("</{}>", tag);

                    (Some(open), Some(processed), Some(close))
                }

                "li" => {
                    // this handles the contents for both ordered and unordered lists

                    let processed = if let Some(text) = node.text {
                        self.inline_process(text, depth+1)
                    } else if !node.contents.is_empty() {
                        self.block_merge(node.contents, depth+1)
                    } else {
                        "".to_string()
                    };


                    let open = if let Some(value) = node.misc {
                        format!("<{} value=\"{}\">", tag, value)
                    } else {
                        format!("<{}>", tag)
                    };

                    let close = format!("</{}>", tag);

                    (Some(open), Some(processed), Some(close))
                }

                _ =>{
                    (None, None, None)
                }
            };

            if let Some(x_opening) = opening {
                let indented = format!(
                    "{:indent$}{}",
                    "",
                    x_opening,
                    indent = (depth * self.indentation)
                );
                result.push(indented)
            }
            if let Some(x_content) = content {
                let indented = format!("{:indent$}{}", "", x_content, indent = 0);
                // already indented in its section, specifically text content
                result.push(indented)
            }
            if let Some(x_closing) = closing {
                let indented = format!(
                    "{:indent$}{}",
                    "",
                    x_closing,
                    indent = (depth * self.indentation)
                );
                result.push(indented)
            }
        }

        result.join("\n")
    }

    fn inline_process(&self, input: String, depth: usize) -> String {
        // convert it into nodes
        let nodes = self.inline_spans(input);

        // merge them together according to the type of node
        let merged = self.inline_merge(nodes);

        // add line breaks if they are required
        // replace "  \n" with "\n<br />\n"
        let line_breaks_added = merged.replace("  \n", "\n<br />\n");

        // add the appropiate indentation to make it readable as raw html
        self.inline_indention(line_breaks_added, depth)
    }

    // this breaks down teh spans
    fn inline_spans(&self, input: String) -> Vec<NodeData> {
        /*

            //em//
            **strong**
            __underline__
            ~~strikethrough~~
            >!spoiler text!< class=md-spoiler-text , hiding it is handled in css, using teh format from Reddit
            ``code``
            <<autolink>>

            yes these are breaking changes with commonmark right abck to daringfireball's one.
            main reason is comformity.
            All of these have a specific opening and closing tag (em/strong from previous implemtations is tricky)
            identifiers dont do multiple jobs, _ was usied in place of * in previous versions


            contents of html are checked for markdown


            [link][]
            ![image]

            Valid:
                Normal:
                    []()
                    [](<>)
                    [text](url "title")
                    [text](<url with spaces>)

                Reference:
                    [identifier]: /url "title"
                    [text][identifier]

        text is subject to markdown
            ye some breaking changes here



            >! spoilered  **strong spoilered //emp strong spoilered// __not underlined** !< __
            >!![]!<

         */

        let mut node_data: Vec<NodeData> = vec![];

        let mut characters: Vec<char> = input.chars().collect();

        let mut index = 0;

        let mut span_closer: Vec<char> = vec![];
        let mut span_start = 0;

        /*
        normal = stuff created by the delimters
        html is any html blocks
        url is anything that matches a url [
        image is like the url but starts with ![
        */

        // marks the start of a span
        let mut span_active = false;
        let mut span_finished = false;
        let mut span_reset = false;

        // for allocating whereitems go
        let mut span_text = true; // defaults to this,
        let mut span_normal = false;
        let mut span_html = false;

        let mut span_normal_type: Option<String> = None;

        let mut html_tag_vec = vec![];
        let mut html_tag_complete = false;

        loop {
            if index >= characters.len() {
                break;
            }

            let character = characters[index];
            let character_next = if (index + 1) < characters.len() {
                Some(characters[index + 1])
            } else {
                None
            };
            let character_last_escape = if (index as isize - 1) > 0 {
                characters[index - 1] == '\\'
            } else {
                false
            };

            // this gets teh first section of a span
            if !span_active {
                match character {
                    // em
                    '/' => {
                        // /
                        if let Some(next) = character_next {
                            if next == '/' {
                                if character_last_escape {
                                    // remove the backslash
                                    characters.remove(index - 1);

                                    // since everything moved to the left so staying still is teh same as the normal +1 for the index
                                    // the -1 here will cancel otut eh +1 later
                                    index -= 1;
                                } else {
                                    span_normal_type = Some("em".to_string());
                                    span_closer = vec!['/', '/'];
                                    span_active = true;
                                    span_normal = true;
                                    span_text = false;
                                }
                            }
                        }
                    }
                    // strong
                    '*' => {
                        // *
                        if let Some(next) = character_next {
                            if next == '*' {
                                if character_last_escape {
                                    // remove the backslash
                                    characters.remove(index - 1);

                                    // since everything moved to the left so staying still is teh same as the normal +1 for the index
                                    // the -1 here will cancel otut eh +1 later
                                    index -= 1;
                                } else {
                                    span_normal_type = Some("strong".to_string());
                                    span_closer = vec!['*', '*'];
                                    span_active = true;
                                    span_normal = true;
                                    span_text = false;
                                }
                            }
                        }
                    }
                    // underline
                    '_' => {
                        // _
                        if let Some(next) = character_next {
                            if next == '_' {
                                if character_last_escape {
                                    // remove the backslash
                                    characters.remove(index - 1);

                                    // since everything moved to the left so staying still is teh same as the normal +1 for the index
                                    // the -1 here will cancel otut eh +1 later
                                    index -= 1;
                                } else {
                                    span_normal_type = Some("u".to_string());
                                    span_closer = vec!['_', '_'];
                                    span_active = true;
                                    span_normal = true;
                                    span_text = false;
                                }
                            }
                        }
                    }
                    // strikethrough
                    '~' => {
                        // ~
                        if let Some(next) = character_next {
                            if next == '~' {
                                if character_last_escape {
                                    // remove the backslash
                                    characters.remove(index - 1);

                                    // since everything moved to the left so staying still is teh same as the normal +1 for the index
                                    // the -1 here will cancel otut eh +1 later
                                    index -= 1;
                                } else {
                                    span_normal_type = Some("s".to_string());
                                    span_closer = vec!['~', '~'];
                                    span_active = true;
                                    span_normal = true;
                                    span_text = false;
                                }
                            }
                        }
                    }
                    // code
                    '`' => {
                        // `
                        if let Some(next) = character_next {
                            if next == '`' {
                                if character_last_escape {
                                    // remove the backslash
                                    characters.remove(index - 1);

                                    // since everything moved to the left so staying still is teh same as the normal +1 for the index
                                    // the -1 here will cancel otut eh +1 later
                                    index -= 1;
                                } else {
                                    span_normal_type = Some("code".to_string());
                                    span_closer = vec!['`', '`'];
                                    span_active = true;
                                    span_normal = true;
                                    span_text = false;
                                }
                            }
                        }
                    }
                    '>' => {
                        // >
                        // only one that breaks teh double mould
                        if let Some(next) = character_next {
                            if next == '!' {
                                if character_last_escape {
                                    // remove the backslash
                                    characters.remove(index - 1);

                                    // since everything moved to the left so staying still is teh same as the normal +1 for the index
                                    // the -1 here will cancel otut eh +1 later
                                    index -= 1;
                                } else {
                                    span_normal_type = Some("spoiler".to_string());
                                    span_closer = vec!['!', '<'];
                                    span_active = true;
                                    span_normal = true;
                                    span_text = false;
                                }
                            }
                        }
                    }
                    '<' => {
                        // set html first
                        span_active = true;
                        span_html = true;
                        span_text = false;

                        if let Some(next) = character_next {
                            // check if autolink
                            match next {
                                '<' => {
                                    // undo the html flag set above
                                    span_html = false;

                                    // set info for the closing
                                    span_normal_type = Some("autolink".to_string());
                                    span_closer = vec!['>', '>'];
                                    span_normal = true;
                                }
                                ' ' | '\t' | '/' | '>' => {
                                    // not a html tag
                                    span_active = false;
                                    span_html = false;
                                }
                                _ => {}
                            }
                        }

                        if span_active && character_last_escape {
                            // reset teh flags
                            span_active = false;
                            span_html = false;
                            span_text = true;
                            span_normal_type = None;
                            span_closer = vec![];
                            span_normal = false;

                            characters.remove(index - 1);

                            // since everything moved to the left so staying still is teh same as the normal +1 for the index
                            // the -1 here will cancel otut eh +1 later
                            index -= 1;
                        }
                    }
                    '[' => {
                        if character_last_escape {
                            // remove the backslash
                            characters.remove(index - 1);

                            // since everything moved to the left so staying still is teh same as the normal +1 for the index
                            // the -1 here will cancel otut eh +1 later
                            index -= 1;
                        } else if let Some((node, offset)) =
                            self.inline_spans_links(&characters[index..], "a".to_string())
                        {
                            // set teh stuff before to be a text
                            let text: String = characters[span_start..index].iter().collect();
                            node_data.push(NodeData {
                                tag: "text".to_string(),
                                misc: None,
                                text: Some(text),
                                contents: vec![],
                            });

                            // add the data to teh array
                            node_data.push(node);
                            index += offset;

                            // set the enw span start
                            span_start = index + 1;
                        }
                    }
                    '!' => {
                        if let Some(next) = character_next {
                            if next == '[' {
                                if character_last_escape {
                                    // remove the backslash
                                    characters.remove(index - 1);

                                    // since everything moved to the left so staying still is teh same as the normal +1 for the index
                                    // the -1 here will cancel otut eh +1 later
                                    index -= 1;
                                } else {
                                    // images require a +1 to the offsets to take into account teh !
                                    if let Some((node, offset)) = self.inline_spans_links(
                                        &characters[(index + 1)..],
                                        "img".to_string(),
                                    ) {
                                        // set teh stuff before to be a text
                                        let text: String =
                                            characters[span_start..index].iter().collect();
                                        node_data.push(NodeData {
                                            tag: "text".to_string(),
                                            misc: None,
                                            text: Some(text),
                                            contents: vec![],
                                        });

                                        // add the data to teh array
                                        node_data.push(node);
                                        index += offset + 1;

                                        // set the new span start
                                        span_start = index + 2;
                                    }
                                }
                            }
                        }
                    }

                    _ => {
                        // do nothing
                    }
                }

                if span_active {
                    // tidy up teh last text span here
                    // bnndle up everythign betweeen span_start and index into a string
                    let text: String = characters[span_start..index].iter().collect();
                    node_data.push(NodeData {
                        tag: "text".to_string(),
                        misc: None,
                        text: Some(text),
                        contents: vec![],
                    });

                    span_start = index;

                    // skip to the next character if html, two if its
                    if span_normal {
                        index += 2;
                    } else {
                        index += 1;
                    }

                    continue;
                }
            }

            // this captures the content of the spans
            if span_active {
                if span_normal {
                    // this aught to be pretty easy, just find the tail end of teh span, byundle it up into NodeData

                    if span_closer[0] == character {
                        if character_last_escape {
                            characters.remove(index - 1);

                            // since everything moved to the left so staying still is teh same as the normal +1 for the index
                            // the -1 here will cancel otut eh +1 later
                            index -= 1;
                        } else if let Some(next) = character_next {
                            if span_closer[1] == next {
                                // mark it as finished
                                span_finished = true;
                            }
                        }
                    }
                }

                if span_html {
                    // of the tag isnt complete then we need to find it
                    if !html_tag_complete {
                        match character {
                            ' ' | '\t' => {
                                html_tag_complete = true;
                            }
                            '/' | '>' => {
                                // check last character, if it was a backslash then remove the backsslash and add the current character
                                if characters[index - 1] == '\\' {
                                    // remove teh slash
                                    // add the bracket
                                    html_tag_vec.pop();
                                    html_tag_vec.push(character);
                                } else {
                                    html_tag_complete = true;
                                }
                            }
                            _ => {
                                // as long as there are no spaces
                                html_tag_vec.push(character);
                            }
                        }
                    }
                    if html_tag_complete {
                        if span_closer.is_empty() {
                            // sets teh
                            let tag: String = html_tag_vec.iter().collect();

                            if self.html_void.contains(&tag) {
                                span_closer = vec!['/', '>'];
                            } else {
                                span_closer = vec!['<', '/'];
                                span_closer.extend(&html_tag_vec);
                                span_closer.push('>');
                            }
                            // reverse it so it can be used easier down below
                            span_closer.reverse();
                        }
                        // using tag_closing find the end of the html span
                        // check if this is the

                        // starts off true
                        let mut matches = true;
                        for (position, closing_char) in span_closer.iter().enumerate() {
                            let index_new: isize = (index as isize) - (position as isize);
                            if index_new < 0 {
                                span_reset = true;
                                matches = false;
                                break;
                            }

                            if &characters[index_new as usize] != closing_char {
                                matches = false;
                                break;
                            }
                        }
                        if matches {
                            span_finished = true;
                        }
                    }
                }
            }

            if character_next.is_none() && !span_finished {
                if span_text {
                    span_finished = true;
                } else {
                    span_reset = true;
                }
            }

            if span_reset {
                // if the first character is < then replace it with &lt;
                if characters[span_start] == '<' {
                    let replacement_char = ['&', 'l', 't', ';'];
                    characters.splice(span_start..=span_start, replacement_char.iter().cloned());
                }

                span_closer = vec![];

                span_active = false;
                span_finished = false;
                span_reset = false; // reset this flag

                span_normal = false;
                span_html = false;
                span_text = true;

                span_normal_type = None;

                html_tag_vec = vec![];
                html_tag_complete = false;

                // reset back to the start of the span
                index = span_start;
                // skip to next character so it wont re-analyise the same set of characters
                index += 1;

                continue;
            }

            if span_finished {
                if span_text {
                    let text: String = characters[span_start..=index].iter().collect();
                    node_data.push(NodeData {
                        tag: "text".to_string(),
                        misc: None,
                        text: Some(text),
                        contents: vec![],
                    });

                    span_start = index;

                    // skip to the next character if html, two if its
                    if span_normal {
                        index += 1;
                    } else {
                        index += 0;
                    }
                }

                if span_normal {
                    // the offset of 2 is to exclude teh delimiter
                    let text_raw: String =
                        characters[(span_start + 2)..=(index - 1)].iter().collect();

                    let span_type = if let Some(span) = span_normal_type {
                        span
                    } else {
                        "text".to_string()
                    };

                    let (text, contents) = match span_type.as_str() {
                        "text" => (Some(text_raw), vec![]),
                        "code" => (Some(text_raw), vec![]),
                        "autolink" => (Some(text_raw), vec![]),
                        _ => (None, self.inline_spans(text_raw)),
                    };

                    node_data.push(NodeData {
                        tag: span_type,
                        misc: None,
                        text,
                        contents,
                    });

                    // skip to after the current delimter
                    span_start = index + 2;

                    // skip to the next character if html, two if its

                    index += 1;
                }

                if span_html {
                    let text: String = characters[span_start..=index].iter().collect();

                    node_data.push(NodeData {
                        tag: "html".to_string(),
                        misc: None,
                        text: Some(text),
                        contents: vec![],
                    });

                    span_start = index + 1;
                }

                span_closer = vec![];

                span_active = false;
                span_finished = false;

                span_normal = false;
                span_html = false;
                span_text = true;

                span_normal_type = None;

                html_tag_vec = vec![];
                html_tag_complete = false;
            }

            index += 1;
        }

        // catch anything at the end
        if span_start < (characters.len() - 1) {
            let text: String = characters[span_start..].iter().collect();
            node_data.push(NodeData {
                tag: "text".to_string(),
                misc: None,
                text: Some(text),
                contents: vec![],
            });
        }

        node_data
    }

    fn inline_spans_links(&self, characters: &[char], tag: String) -> Option<(NodeData, usize)> {
        let references = self.references.clone();

        // index starts a t 1 as first character is alwways the opener [
        let mut index = 1;

        let mut block_first = vec![];

        let mut link_type = None;
        loop {
            if index >= characters.len() {
                break;
            }

            let character = characters[index];
            let character_next = if (index + 1) < characters.len() {
                Some(characters[index + 1])
            } else {
                None
            };
            let character_last_escape = if (index as isize - 1) > 0 {
                characters[index - 1] == '\\'
            } else {
                false
            };

            match character {
                ']' => {
                    if character_last_escape {
                        block_first.pop();
                        block_first.push(character)
                    } else {
                        // find the type of link
                        if let Some(next) = character_next {
                            match next {
                                '(' => {
                                    link_type = Some("classic".to_string());
                                    // going to need to take a look at the contents
                                    index += 2;
                                }
                                '[' => {
                                    link_type = Some("reference_named".to_string());
                                    index += 2;
                                }
                                ' ' | '\t' | '\n' => {
                                    // check if the link_text is the same as the reference links in self.references
                                    link_type = Some("reference_anon".to_string());
                                }
                                _ => {
                                    // if it dosent match above its not a link
                                    return None;
                                }
                            }
                        } else {
                            // set up to test if its teh anon type
                            link_type = Some("reference_anon".to_string());
                        }

                        // regardless break it here
                        break;
                    }
                }

                _ => block_first.push(character),
            }

            index += 1;
        }

        // deal with teh anon references
        if let Some(type_) = &link_type {
            // only concerned with teh anon_test one this time
            if let "reference_anon" = type_.as_str() {
                let reference: String = block_first.iter().collect();

                return match references.get(&reference) {
                    Some(link_data) => {
                        let url = link_data.url.clone();
                        // reference us used as the contents if its an anon one
                        let contents = self.inline_spans(reference);

                        let node = NodeData {
                            tag,
                            misc: link_data.title.clone(),
                            // text here is used as the url
                            text: Some(url),
                            contents,
                        };

                        Some((node, index))
                    }
                    None => {
                        // not a link
                        None
                    }
                };
            }
        }

        let mut block_second = vec![];
        let mut angle_brackets = false;
        loop {
            if index >= characters.len() {
                break;
            }

            let character = characters[index];
            let character_last_escape = if (index as isize - 1) > 0 {
                characters[index - 1] == '\\'
            } else {
                false
            };

            // do stuff here

            if let Some(type_) = &link_type {
                match type_.as_str() {
                    "classic" => {
                        match character {
                            '<' => {
                                if block_second.is_empty() {
                                    angle_brackets = true;
                                }

                                // add it regardless, can be easially removed later
                                block_second.push(character)
                            }
                            ' ' | '\t' => {
                                if angle_brackets {
                                    block_second.push(character)
                                } else if block_second.is_empty() {
                                    // do nothing,
                                } else {
                                    // next loop wills tart on teh enxt character
                                    index += 1;
                                    // its teh end of the url
                                    break;
                                }
                            }
                            ')' => {
                                if angle_brackets {
                                    block_second.push(character)
                                } else if character_last_escape {
                                    block_second.pop();
                                    block_second.push(character);
                                } else {
                                    // this marks the classic link as finished
                                    // so make it and finish early

                                    let text_raw: String = block_first.iter().collect();
                                    let contents = self.inline_spans(text_raw);
                                    let url: String = block_second.iter().collect();
                                    let node = NodeData {
                                        tag,
                                        misc: None,
                                        text: Some(url),
                                        contents,
                                    };

                                    return Some((node, index));
                                }
                            }
                            '\n' => {
                                // all inline links must be on the one line, not split over multiple
                                // saves a lot of complexity
                                return None;
                            }
                            '>' => {
                                if angle_brackets {
                                    // marks teh end of the span
                                    // however if last char is a backslash its ignored
                                    if character_last_escape {
                                        block_second.pop();
                                        block_second.push(character);
                                    } else {
                                        // remove the first char which is a <
                                        block_second.remove(0);
                                        break;
                                    }
                                } else {
                                    block_second.push(character)
                                }
                            }
                            _ => block_second.push(character),
                        }
                    }
                    "reference_named" => match character {
                        ']' => {
                            if character_last_escape {
                                block_second.pop();
                                block_second.push(character);
                            } else {
                                break;
                            }
                        }
                        _ => block_second.push(character),
                    },
                    _ => {
                        // only classic and reference_named should show up
                    }
                }
            }

            index += 1;
        }

        // tidy up the references
        if let Some(type_) = &link_type {
            // not for classic
            if let "reference_named" = type_.as_str() {
                let reference: String = block_second.iter().collect();

                return match references.get(&reference) {
                    Some(link_data) => {
                        let text_raw: String = block_first.iter().collect();
                        let contents = self.inline_spans(text_raw);

                        let node = NodeData {
                            tag,
                            misc: link_data.title.clone(),
                            // text here is used as the url
                            text: Some(link_data.url.clone()),
                            contents,
                        };

                        Some((node, index))
                    }
                    None => {
                        // not a link
                        None
                    }
                };
            }
        }

        //
        /*
        now just left with classic

        // tehsea re the three general patterns to find
        [](<>)
        [](/url )
        [](/url "title")
         */

        // this handles the title
        let mut block_third = vec![];
        let mut delimiter = (' ', false, false);
        loop {
            if index >= characters.len() {
                break;
            }

            let character = characters[index];
            let character_last_escape = if (index as isize - 1) > 0 {
                characters[index - 1] == '\\'
            } else {
                false
            };

            match character {
                // manage delimiter
                '\'' | '"' => {
                    if !delimiter.1 {
                        // set the delimiter
                        delimiter = (character, true, false);
                    } else if delimiter.0 == character {
                        // check if last character was an escape
                        if character_last_escape {
                            block_third.pop();
                            block_third.push(character);
                        } else {
                            // mark it closed
                            delimiter = (character, true, true);
                        }
                    } else {
                        block_third.push(character);
                    }
                }

                //
                ')' => {
                    if delimiter.1 && !delimiter.2 {
                        // if delimiter is open then add it to teh array
                        block_third.push(character);
                    } else {
                        // else chack if its escaped
                        if character_last_escape {
                            block_third.pop();
                            block_third.push(character);
                        } else {
                            // this marks the classic link as finished
                            // so make it and finish early

                            let text_raw: String = block_first.iter().collect();
                            let contents = self.inline_spans(text_raw);

                            let url: String = block_second.iter().collect();

                            let title = if block_third.is_empty() {
                                None
                            } else {
                                let title_tmp: String = block_third.iter().collect();
                                Some(title_tmp)
                            };

                            let node = NodeData {
                                tag,
                                misc: title,
                                text: Some(url),
                                contents,
                            };

                            return Some((node, index));
                        }
                    }
                }

                ' ' | '\t' => {
                    if delimiter.1 && !delimiter.2 {
                        // if delimiter is open then add it to teh array
                        block_third.push(character);
                    } else {
                        // do nothing
                    }
                }
                _ => {
                    block_third.push(character);
                }
            }

            index += 1;
        }

        // if its gone to teh end without being closed out then its invalid

        None
    }

    fn inline_merge(&self, nodes: Vec<NodeData>) -> String {
        let mut result: Vec<String> = vec![];

        for node in nodes {
            let tag = node.tag.as_str();
            let (opening, content, closing) = match tag {
                // html  and text just gets passed straight through
                "html" | "text" => {
                    if let Some(text) = node.text {
                        result.push(text);
                    }
                    (None, None, None)
                }
                // treat this basically the same as html above
                "autolink" => {
                    if let Some(text) = node.text {
                        let (cleaned, mail, tel) = if text.starts_with("mailto:") {
                            (text.replacen("mailto:", "", 1), true, false)
                        } else if text.starts_with("MAILTO:") {
                            (text.replacen("MAILTO:", "", 1), true, false)
                        } else if text.starts_with("tel:") {
                            (text.replacen("tel:", "", 1), false, true)
                        } else if text.starts_with("TEL:") {
                            (text.replacen("TEL:", "", 1), false, true)
                        } else {
                            (text, false, false)
                        };

                        // check if it contains @
                        let formatted = if cleaned.contains('@') || mail {
                            format!("<a target='_blank' rel='noopener noreferrer' href='mailto:{}'>{}</a>", &cleaned, &cleaned)
                        } else if tel {
                            format!(
                                "<a target='_blank' rel='noopener noreferrer' href='tel:{}'>{}</a>",
                                &cleaned, &cleaned
                            )
                        } else {
                            format!(
                                "<a target='_blank' rel='noopener noreferrer' href='{}'>{}</a>",
                                &cleaned, &cleaned
                            )
                        };

                        result.push(formatted);
                    }
                    (None, None, None)
                }

                // set tags and no recursion
                "code" => {
                    if let Some(text) = node.text {
                        // no attributes or anything recursive
                        let open = format!("<{}>", tag);
                        let close = format!("</{}>", tag);

                        (Some(open), Some(text), Some(close))
                    } else {
                        (None, None, None)
                    }
                }

                // recursive
                "em" | "strong" | "u" | "s" => {
                    let processed = self.inline_merge(node.contents);
                    let open = format!("<{}>", tag);
                    let close = format!("</{}>", tag);
                    (Some(open), Some(processed), Some(close))
                }

                // spoiler is a span with a class of class="md-spoiler" on it as there is no html element for spoilers and must be done using css
                "spoiler" => {
                    let processed = self.inline_merge(node.contents);
                    let open = "<span class='md-spoiler'>".to_string();
                    let close = "</span>".to_string();
                    (Some(open), Some(processed), Some(close))
                }

                "a" => {
                    let url = node.text.unwrap_or_default();
                    let open = if let Some(title) = node.misc {
                        format!(
                            "<a target='_blank' rel='noopener noreferrer' href='{}' title='{}'>",
                            url, title
                        )
                    } else {
                        format!(
                            "<a target='_blank' rel='noopener noreferrer' href='{}'>",
                            url
                        )
                    };

                    let processed = self.inline_merge(node.contents);
                    let close = "</a>".to_string();

                    (Some(open), Some(processed), Some(close))
                }

                "img" => {
                    let url = node.text.unwrap_or_default();
                    let alt = self.inline_merge(node.contents);
                    let open = if let Some(title) = node.misc {
                        format!("<img src='{}' alt='{}' title='{}' />", url, alt, title)
                    } else {
                        format!("<img src='{}' alt='{}' />", url, alt)
                    };

                    (Some(open), None, None)
                }

                _ => (None, None, None),
            };

            if let Some(data) = opening {
                result.push(data)
            }
            if let Some(data) = content {
                result.push(data)
            }
            if let Some(data) = closing {
                result.push(data)
            }
        }

        result.join("")
    }

    fn inline_indention(&self, input: String, depth: usize) -> String {
        let mut result: Vec<String> = vec![];

        let lines = input.split('\n').collect::<Vec<_>>();
        for line in lines {
            let indented = format!(
                "{:indent$}{}",
                "",
                line,
                indent = (depth * self.indentation)
            );
            result.push(indented);
        }

        result.join("\n")
    }
}
