// module that does the conversion
use bfom_lib::{
    utilities::processing::{copy_css, file_find_markdown, uppercase_first_letter},
    Converter,
};

use std::{fs, io::Error, path::Path, path::PathBuf};

fn main() -> Result<(), Error> {
    println!("Generating blog");

    let converter = Controller::new();
    converter.run()?;

    println!("Finished Generating blog");
    Ok(())
}

#[derive(Debug)]
pub struct Controller {
    html_void: Vec<String>,
    indentation: usize,
    src: PathBuf,
    dest: PathBuf,
    templates_order: Vec<Template>,
}

#[derive(Debug)]
enum Template {
    Adjacent,
    Blog(PathBuf),
    BlogFolder(PathBuf),
    General(PathBuf),
}

impl Default for Controller {
    fn default() -> Self {
        Self::new()
    }
}

fn vec_str_to_string(input: &[&str]) -> Vec<String> {
    let mut result = vec![];

    for item in input.iter() {
        result.push(item.to_string())
    }

    result
}

// uses templates to get the desired result
impl Controller {
    // takes the external config if provided and returns an instance of itself
    pub fn new() -> Self {
        // configs are set here, no override
        Controller {
            src: "./src".parse().unwrap(),
            dest: "./build".parse().unwrap(),

            indentation: 2,

            html_void: vec_str_to_string(&["area", "base", "br", "col", "command", "embed", "hr", "img", "input", "keygen", "link", "meta", "param", "source", "track", "wbr"]),

            // pass in both  teh order and the path to teh templates
            templates_order: vec![Template::Adjacent, Template::BlogFolder("./src/templates/folder.html".parse().unwrap()), Template::Blog("./src/templates/blog.html".parse().unwrap()), Template::General("./src/templates/general.html".parse().unwrap())],
        }
    }

    // this manages the conversion
    pub fn run(&self) -> Result<(), Error> {
        // wipe the existing dest folder
        if self.dest.is_dir() {
            fs::remove_dir_all(&self.dest)?;
        }

        // get all teh files in teh specified folder
        let files = file_find_markdown(&self.src)?;

        copy_css(&self.src, &self.dest)?;

        //set up the converter
        let mut converter = Converter::new(self.html_void.clone(), self.indentation);

        let numbers = self.get_blog_numbers();

        // iterate through the files  replacing the extension, the src to dest
        for file in files {
            let src = file.clone();

            let mut dest = if let Ok(stripped) = file.clone().strip_prefix(&self.src) {
                Path::new(&self.dest).join(stripped)
            } else {
                // if the prefix cannot be stripped then put in the output files beside teh in
                // though considering that its based of of src this should never be called
                file.clone()
            };
            dest.set_extension("html");

            // we know the src folders exist, but check if the dest ones do
            if let Some(parent) = dest.parent() {
                fs::create_dir_all(parent)?
            }

            // get teh template, if applicable
            let (template, depth, template_type) = self.template_get(&src);

            // send it to the converter function
            let processed = converter.convert_file(&src, depth)?;

            // merge template and processed
            let merged = self.template_merge(&src, processed, template, template_type, &numbers);

            // save it
            fs::write(&dest, merged)?;
        }

        Ok(())
    }

    fn template_get(&self, input: &Path) -> (Option<String>, usize, Template) {
        for item in &self.templates_order {
            match item {
                Template::Adjacent => {
                    // check for <path>/<name>.html
                    let mut input_test = input.to_path_buf();
                    input_test.set_extension("html");

                    if let Ok(template_file) = fs::read_to_string(input_test) {
                        return (Some(template_file), 1, Template::Adjacent);
                    }
                }
                Template::General(template) => {
                    // see if general is set in teh config
                    // then see if that files actually exists
                    if let Ok(template_file) = fs::read_to_string(template) {
                        return (Some(template_file), 1, Template::General(Default::default()));
                    }
                }
                Template::Blog(template) => {
                    if !input.to_str().unwrap_or_default().contains("blog") {
                        continue;
                    }

                    if let Ok(template_file) = fs::read_to_string(template) {
                        return (Some(template_file), 1, Template::Blog(Default::default()));
                    }
                }

                Template::BlogFolder(template) => {
                    if !input.to_str().unwrap_or_default().contains("blog") {
                        continue;
                    }

                    let path = input.to_path_buf();
                    // yes this looks cursed to me too

                    // using soem of my original code
                    if let Some(parent) = path.parent() {
                        let mut to_be_tested = PathBuf::from(parent);
                        to_be_tested.set_extension("md");

                        // test if there is a file with teh same name as the folder this resides in
                        if fs::read_to_string(to_be_tested).is_ok() {
                            if let Ok(template_file) = fs::read_to_string(template) {
                                return (Some(template_file), 1, Template::BlogFolder(Default::default()));
                            }
                        }
                    }
                }
            }
        }

        // fallback to none
        (None, 0, Template::General(Default::default()))
    }
    fn template_merge(&self, input: &Path, processed: String, template: Option<String>, template_type: Template, numbers: &[u16]) -> String {
        if let Some(mut template_string) = template {
            // use the file name for the title
            let mut input_test = input.to_path_buf();
            input_test.set_extension("");

            let title: String = if let Some(file_name) = input_test.file_name() {
                if let Some(result) = file_name.to_str() {
                    uppercase_first_letter(result)
                } else {
                    Default::default()
                }
            } else {
                Default::default()
            };
            template_string = template_string.replace("{title}", &title).replace("{body}", &processed);

            // blog stuff
            if let Template::Blog(_) = template_type {
                if let Some(file_stem) = input.file_stem() {
                    // all blog items are basically guarenteed to be in this
                    //let numbers = vec![1, "2", "3","4","5","7"];

                    let current = file_stem.to_string_lossy().parse::<u16>().unwrap_or(1);
                    let index = numbers.iter().position(|&x| x == current).unwrap_or(0);

                    let (first, prev) = if index > 0 {
                        (numbers[0].to_string(), numbers[index - 1].to_string())
                    } else {
                        ("#".to_string(), "#".to_string())
                    };
                    let (next, last) = if index < (numbers.len() - 1) {
                        (numbers[index + 1].to_string(), numbers[numbers.len() - 1].to_string())
                    } else {
                        ("#".to_string(), "#".to_string())
                    };

                    template_string = template_string
                        .replace("{first}", &first)
                        .replace("{prev}", &prev)
                        .replace("{next}", &next)
                        .replace("{last}", &last);
                };
            }

            if let Template::BlogFolder(_) = template_type {
                let folder = if let Some(parent) = input_test.parent() {
                    if let Some(parent_name) = parent.file_name() {
                        if let Some(parent_unwrapped) = parent_name.to_str() {
                            parent_unwrapped
                        } else {
                            Default::default()
                        }
                    } else {
                        Default::default()
                    }
                } else {
                    Default::default()
                };

                // replace teh title and the body
                template_string = template_string.replace("{folder}", folder);
            }

            template_string
        } else {
            // no template, just return teh processed right back
            processed
        }
    }

    fn get_blog_numbers(&self) -> Vec<u16> {
        let mut result = vec![];

        // go to ./src/blog
        // iterate through teh files
        // if a filename is a number then add it to teh array

        let mut blog = PathBuf::from(&self.src);
        blog.push("blog");

        if let Ok(entries) = fs::read_dir(blog) {
            for entry in entries.flatten() {
                let path = entry.path();
                if path.is_dir() {
                    continue;
                } else {
                    // check if the name is a number
                    if let Some(file_stem) = path.file_stem() {
                        if let Ok(number) = file_stem.to_string_lossy().parse::<u16>() {
                            result.push(number)
                        }
                    }
                }
            }
        }

        // sort array
        result.sort_unstable();

        result
    }
}
