use bevy_math::Vec3;
use bevy_reflect::Reflect;

/// Defines how the cloth will compute sticks from mesh indices.
#[derive(Debug, Copy, Clone, Reflect)]
pub enum StickGeneration {
    /// 3 sticks will be generated by triangle
    Triangles,
    /// 2 sticks will be generated by triangle, following the actual quad edges
    Quads,
}

/// Defines the target length of cloth sticks
#[derive(Debug, Copy, Clone, Reflect)]
pub enum StickLen {
    /// The target length will be the actual distance between the vertices
    Auto,
    /// Custom target length
    Fixed(f32),
    /// Same as [`StickLen::Auto`] with a custom additional offset
    Offset(f32),
    /// Same as [`StickLen::Auto`] with a custom coefficient
    Coefficient(f32),
}

impl Default for StickGeneration {
    fn default() -> Self {
        Self::Quads
    }
}

impl Default for StickLen {
    fn default() -> Self {
        Self::Auto
    }
}

impl StickLen {
    /// Retrieves the stick length from the two points it connects
    #[must_use]
    pub fn get_stick_len(&self, point_a: Vec3, point_b: Vec3) -> f32 {
        match self {
            Self::Auto => point_a.distance(point_b),
            Self::Fixed(v) => *v,
            Self::Offset(offset) => point_a.distance(point_b) + offset,
            Self::Coefficient(coeff) => point_a.distance(point_b) * coeff,
        }
    }
}
