use bevy_ecs::prelude::{Component, ReflectComponent};
use bevy_math::Vec3;
use bevy_reflect::Reflect;

/// Cloth physics configuration.
///
/// Used as a resource, it is used as a global configuration for every cloth entity.
/// Used as a component on a cloth entity, it overrides the global values for that cloth.
#[cfg_attr(feature = "debug", derive(bevy_inspector_egui::Inspectable))]
#[derive(Debug, Clone, Component, Reflect)]
#[reflect(Component)]
pub struct ClothConfig {
    /// Custom gravity, classic (0, -9.81, 0) is used by default
    pub gravity: Vec3,
    /// Custom friction to apply to velocity. Useful to reduce the elasticity of a cloth.
    ///
    /// Note: The friction is not applied to external accelerations like gravity and winds
    /// Note: will be clamped between 0.0 and 1.0
    pub friction: f32,
    /// Sets the number of sticks computation iteration.
    /// The higher the value, the more precision and less elasticity for the sticks but the cost is increased
    pub sticks_computation_depth: u8,
}

impl ClothConfig {
    /// Default Y value for gravity
    pub const DEFAULT_GRAVITY: f32 = -9.81;

    #[must_use]
    #[inline]
    pub(crate) fn friction_coefficient(&self) -> f32 {
        1.0 - self.friction.clamp(0.0, 1.0)
    }

    /// Initializes a cloth config with no gravity force
    #[must_use]
    #[inline]
    pub fn no_gravity() -> Self {
        Self {
            gravity: Vec3::ZERO,
            ..Default::default()
        }
    }
}

impl Default for ClothConfig {
    fn default() -> Self {
        Self {
            gravity: Vec3::Y * Self::DEFAULT_GRAVITY,
            friction: 0.05,
            sticks_computation_depth: 5,
        }
    }
}
