use bevy::prelude::*;

use bevy_ui_navigation::systems::{
    default_gamepad_input, default_keyboard_input, default_mouse_input, InputMapping,
};
use bevy_ui_navigation::{Focusable, NavEvent, NavigationPlugin};

/// This example illustrates how to mark buttons as focusable and let
/// NavigationPlugin figure out how to go from one to another.
fn main() {
    App::new()
        .add_plugins(DefaultPlugins)
        // 1: Add the NavigationPlugin
        .add_plugin(NavigationPlugin)
        .init_resource::<ButtonMaterials>()
        .init_resource::<InputMapping>()
        .add_startup_system(setup)
        .add_system(button_system)
        .add_system(default_keyboard_input)
        .add_system(default_gamepad_input)
        .add_system(default_mouse_input)
        .add_system(print_nav_events)
        .run();
}

struct ButtonMaterials {
    normal: Color,
    focused: Color,
    active: Color,
}

impl Default for ButtonMaterials {
    fn default() -> Self {
        ButtonMaterials {
            normal: Color::DARK_GRAY,
            focused: Color::ORANGE_RED,
            active: Color::GOLD,
        }
    }
}

#[allow(clippy::type_complexity)]
fn button_system(
    button_materials: Res<ButtonMaterials>,
    mut interaction_query: Query<
        (&Focusable, &mut UiColor),
        (Changed<Focusable>, With<Button>),
    >,
) {
    for (focus_state, mut material) in interaction_query.iter_mut() {
        if focus_state.is_focused() {
            *material = button_materials.focused.into();
        } else if focus_state.is_active() {
            *material = button_materials.active.into();
        } else {
            *material = button_materials.normal.into();
        }
    }
}
fn print_nav_events(mut events: EventReader<NavEvent>) {
    for event in events.iter() {
        println!("{:?}", event);
    }
}

macro_rules! xy {
    ($x:expr, $y:expr) => {
        Vec2::new($x as f32, $y as f32)
    };
}
fn setup(mut commands: Commands, button_materials: Res<ButtonMaterials>) {
    // ui camera
    commands.spawn_bundle(UiCameraBundle::default());
    let positions = [
        xy!(10, 10),
        xy!(15, 50),
        xy!(20, 90),
        xy!(30, 10),
        xy!(35, 50),
        xy!(40, 90),
        xy!(60, 10),
        xy!(55, 50),
        xy!(50, 90),
    ];
    commands
        .spawn_bundle(NodeBundle {
            style: Style {
                position_type: PositionType::Absolute,
                size: Size::new(Val::Percent(100.0), Val::Percent(100.0)),
                ..Default::default()
            },
            ..Default::default()
        })
        .with_children(|commands| {
            for pos in positions {
                spawn_button(pos, commands, &button_materials);
            }
        });
}
fn spawn_button(position: Vec2, commands: &mut ChildBuilder, button_materials: &ButtonMaterials) {
    let position = Rect {
        left: Val::Percent(position.x),
        top: Val::Percent(position.y),
        ..Default::default()
    };
    commands
        .spawn_bundle(ButtonBundle {
            style: Style {
                size: Size::new(Val::Px(95.0), Val::Px(65.0)),
                position,
                position_type: PositionType::Absolute,
                ..Default::default()
            },
            color: button_materials.normal.into(),
            ..Default::default()
        })
        // 2. Add the `Focusable` component to the navigable Entity
        .insert(Focusable::default());
}
