use bevy::prelude::*;
use bevy_inspector_egui::{Inspectable, InspectorPlugin};
use bevy_mod_picking::{
    InteractablePickingPlugin, PickableBundle, PickingCameraBundle, PickingPlugin,
};

#[derive(Inspectable, Default)]
struct Inspector {
    #[inspectable(deletable = false)]
    active: Option<Entity>,
}

fn main() {
    App::build()
        .insert_resource(Msaa { samples: 4 })
        .add_plugins(DefaultPlugins)
        // add bevy_mod_picking plugins
        .add_plugin(PickingPlugin)
        .add_plugin(InteractablePickingPlugin)
        // add inspector
        .add_plugin(InspectorPlugin::<Inspector>::new())
        .add_system_to_stage(
            CoreStage::PostUpdate,
            maintain_inspected_entities
                .system()
                .after(bevy_mod_picking::PickingSystem::Focus),
        )
        .add_startup_system(setup.system())
        .run();
}

fn maintain_inspected_entities(
    mut inspector: ResMut<Inspector>,
    query: Query<(Entity, &Interaction), Changed<Interaction>>,
) {
    let entity = query
        .iter()
        .filter(|(_, interaction)| matches!(interaction, Interaction::Clicked))
        .map(|(entity, _)| entity)
        .next();

    if let Some(entity) = entity {
        if inspector.active == Some(entity) {
            inspector.active = None;
        } else {
            inspector.active = Some(entity);
        }
    }
}

/// set up a simple 3D scene
fn setup(
    mut commands: Commands,
    mut meshes: ResMut<Assets<Mesh>>,
    mut materials: ResMut<Assets<StandardMaterial>>,
) {
    // add entities to the world
    // camera
    commands
        .spawn_bundle(PerspectiveCameraBundle {
            transform: Transform::from_matrix(Mat4::face_toward(
                Vec3::new(-3.0, 5.0, 8.0),
                Vec3::new(0.0, 0.0, 0.0),
                Vec3::new(0.0, 1.0, 0.0),
            )),
            ..Default::default()
        })
        .insert_bundle(PickingCameraBundle::default());
    // plane
    commands
        .spawn_bundle(PbrBundle {
            mesh: meshes.add(Mesh::from(shape::Plane { size: 6.0 })),
            material: materials.add(Color::rgb(1.0, 1.0, 1.0).into()),
            ..Default::default()
        })
        .insert_bundle(PickableBundle::default());
    // cube
    commands
        .spawn_bundle(PbrBundle {
            mesh: meshes.add(Mesh::from(shape::Cube { size: 1.0 })),
            transform: Transform::from_xyz(-1.0, 1.0, 0.0),
            material: materials.add(Color::VIOLET.into()),
            ..Default::default()
        })
        .insert_bundle(PickableBundle::default());
    // sphere
    commands
        .spawn_bundle(PbrBundle {
            mesh: meshes.add(Mesh::from(shape::Icosphere {
                subdivisions: 20,
                radius: 0.5,
            })),
            transform: Transform::from_xyz(0.8, 1.5, 2.5),
            material: materials.add(Color::MIDNIGHT_BLUE.into()),
            ..Default::default()
        })
        .insert_bundle(PickableBundle::default());
    // light
    commands.spawn_bundle(LightBundle {
        transform: Transform::from_xyz(4.0, 8.0, 4.0),
        ..Default::default()
    });
}
