struct View {
    view_proj: mat4x4<f32>;
    view: mat4x4<f32>;
    inverse_view: mat4x4<f32>;
    projection: mat4x4<f32>;
    world_position: vec3<f32>;
    near: f32;
    far: f32;
    width: f32;
    height: f32;
};

struct Cuboid {
    min: vec4<f32>;
    max: vec4<f32>;
    color: vec4<f32>;
};

struct Cuboids {
    data: array<Cuboid>;
};

[[group(0), binding(0)]]
var<uniform> view: View;

[[group(1), binding(0)]]
var<storage> cuboids: Cuboids;

struct VertexOutput {
    [[builtin(position)]] clip_position: vec4<f32>;
    [[location(0)]] color: vec4<f32>;
};

[[stage(vertex)]]
fn vertex([[builtin(vertex_index)]] vertex_index: u32) -> VertexOutput {
    let instance_index = vertex_index >> 3u;
    let cuboid = cuboids.data[instance_index];

    let cuboid_center = (cuboid.min + cuboid.max) / 2.0;
    let local_camera_pos = view.world_position - cuboid_center.xyz;

    let mirror_mask =
        u32(local_camera_pos.x < 0.0) |
        u32(local_camera_pos.y < 0.0) << 1u |
        u32(local_camera_pos.z < 0.0) << 2u;
    let visible_vertex_index = vertex_index ^ mirror_mask;
    var cube_corner: vec3<f32> = vec3<f32>(
        f32(visible_vertex_index & 0x1u),
        f32((visible_vertex_index & 0x2u) >> 1u),
        f32((visible_vertex_index & 0x4u) >> 2u),
    );
    let world_position = cube_corner * cuboid.max.xyz + (1.0 - cube_corner) * cuboid.min.xyz;
    let world_position = vec4<f32>(world_position, 1.0);

    var out: VertexOutput;
    out.clip_position = view.view_proj * world_position;
    out.color = cuboid.color;
    return out;
}

struct FragmentInput {
    [[location(0)]] color: vec4<f32>;
};

[[stage(fragment)]]
fn fragment(in: FragmentInput) -> [[location(0)]] vec4<f32> {
    return in.color;
}
