use super::draw::DrawCuboids;
use super::extract::{extract_cuboids, extract_cuboids_phase};
use super::pass_node::{self, CuboidsPassNode};
use super::pipeline::{CuboidsPipeline, CUBES_SHADER_HANDLE};
use super::prepare::{prepare_cuboids, BufferCache};
use super::queue::{queue_cuboids, CuboidsPhaseItem};

use bevy::prelude::*;
use bevy::{
    core_pipeline::draw_3d_graph,
    render::{
        render_graph::RenderGraph,
        render_phase::{AddRenderCommand, DrawFunctions},
        RenderApp, RenderStage,
    },
};

/// Renders the [`Cuboids`](crate::Cuboids) component using the "vertex pulling" technique.
pub struct VertexPullingRenderPlugin;

impl Plugin for VertexPullingRenderPlugin {
    fn build(&self, app: &mut App) {
        app.world.resource_mut::<Assets<Shader>>().set_untracked(
            CUBES_SHADER_HANDLE,
            Shader::from_wgsl(include_str!("vertex_pulling.wgsl")),
        );

        let render_app = app.sub_app_mut(RenderApp);

        render_app
            .init_resource::<DrawFunctions<CuboidsPhaseItem>>()
            .add_render_command::<CuboidsPhaseItem, DrawCuboids>()
            .init_resource::<CuboidsPipeline>()
            .init_resource::<BufferCache>()
            .add_system_to_stage(RenderStage::Extract, extract_cuboids_phase)
            .add_system_to_stage(RenderStage::Extract, extract_cuboids)
            .add_system_to_stage(RenderStage::Prepare, prepare_cuboids)
            .add_system_to_stage(RenderStage::Queue, queue_cuboids);

        let cuboids_pass_node = CuboidsPassNode::new(&mut render_app.world);
        let mut graph = render_app.world.resource_mut::<RenderGraph>();
        let draw_3d_graph = graph.get_sub_graph_mut(draw_3d_graph::NAME).unwrap();
        draw_3d_graph.add_node(pass_node::CUBES_PASS, cuboids_pass_node);
        draw_3d_graph
            .add_node_edge(pass_node::CUBES_PASS, draw_3d_graph::node::MAIN_PASS)
            .unwrap();
        draw_3d_graph
            .add_slot_edge(
                draw_3d_graph.input_node().unwrap().id,
                draw_3d_graph::input::VIEW_ENTITY,
                pass_node::CUBES_PASS,
                CuboidsPassNode::IN_VIEW,
            )
            .unwrap();
    }
}
