use super::prepare::BufferCache;
use super::queue::CuboidsPhaseItem;
use crate::component::*;

use bevy::{
    prelude::*,
    render::{
        camera::{ActiveCamera, Camera3d},
        render_phase::RenderPhase,
        RenderWorld,
    },
};

pub(crate) fn extract_cuboids_phase(
    mut commands: Commands,
    active_3d: Res<ActiveCamera<Camera3d>>,
) {
    if let Some(entity) = active_3d.get() {
        commands
            .get_or_spawn(entity)
            .insert(RenderPhase::<CuboidsPhaseItem>::default());
    }
}

pub(crate) fn extract_cuboids(
    mut commands: Commands,
    mut render_world: ResMut<RenderWorld>,
    cuboids: Query<(
        Entity,
        &Cuboids,
        Option<&CuboidsMask>,
        Or<(
            Added<Cuboids>,
            Changed<Cuboids>,
            Added<CuboidsMask>,
            Changed<CuboidsMask>,
        )>,
    )>,
) {
    let mut buffer_cache = render_world.resource_mut::<BufferCache>();
    for (entity, cuboids, maybe_mask, needs_update) in cuboids.iter() {
        let mut entity_cmd = commands.get_or_spawn(entity);
        if needs_update {
            let render_cuboids = if let Some(mask) = maybe_mask {
                assert_eq!(mask.bitmask.len(), cuboids.instances.len());
                Cuboids::new(
                    mask.bitmask
                        .iter_zeros()
                        .map(|index| cuboids.instances[index])
                        .collect(),
                )
            } else {
                cuboids.clone()
            };
            entity_cmd.insert(render_cuboids);
            // Buffer cache will get filled in RenderState::Prepare.
        } else {
            buffer_cache.keep_alive(entity);
        }
    }
    buffer_cache.cull_entities();
}
