use super::{pipeline::CuboidsPipeline, prepare::BufferCache};

use bevy::{
    ecs::system::{lifetimeless::SRes, SystemParamItem},
    pbr::SetShadowViewBindGroup,
    prelude::*,
    render::{
        render_phase::{
            EntityRenderCommand, PhaseItem, RenderCommand, RenderCommandResult, TrackedRenderPass,
        },
        render_resource::{IndexFormat, PipelineCache},
    },
};

pub(crate) type DrawCuboids = (
    SetCuboidsPipeline,
    SetShadowViewBindGroup<0>,
    SetGpuCuboidBuffersBindGroup<1>,
    DrawVertexPulledCuboids,
);

pub(crate) struct SetCuboidsPipeline;

impl<P: PhaseItem> RenderCommand<P> for SetCuboidsPipeline {
    type Param = (SRes<PipelineCache>, SRes<CuboidsPipeline>);

    #[inline]
    fn render<'w>(
        _view: Entity,
        _item: &P,
        params: SystemParamItem<'w, '_, Self::Param>,
        pass: &mut TrackedRenderPass<'w>,
    ) -> RenderCommandResult {
        let (pipeline_cache, cuboids_pipeline) = params;
        if let Some(pipeline) = pipeline_cache
            .into_inner()
            .get_render_pipeline(cuboids_pipeline.pipeline_id)
        {
            pass.set_render_pipeline(pipeline);
            RenderCommandResult::Success
        } else {
            RenderCommandResult::Failure
        }
    }
}

pub(crate) struct SetGpuCuboidBuffersBindGroup<const I: usize>;

impl<const I: usize> EntityRenderCommand for SetGpuCuboidBuffersBindGroup<I> {
    type Param = SRes<BufferCache>;

    #[inline]
    fn render<'w>(
        _view: Entity,
        item: Entity,
        buffer_cache: SystemParamItem<'w, '_, Self::Param>,
        pass: &mut TrackedRenderPass<'w>,
    ) -> RenderCommandResult {
        let buffers = buffer_cache.into_inner().get_buffers(item).unwrap();
        pass.set_bind_group(I, &buffers.instance_buffer_bind_group, &[]);

        RenderCommandResult::Success
    }
}

pub(crate) struct DrawVertexPulledCuboids;

impl EntityRenderCommand for DrawVertexPulledCuboids {
    type Param = SRes<BufferCache>;

    #[inline]
    fn render<'w>(
        _view: Entity,
        item: Entity,
        buffer_cache: SystemParamItem<'w, '_, Self::Param>,
        pass: &mut TrackedRenderPass<'w>,
    ) -> RenderCommandResult {
        let buffers = buffer_cache.into_inner().get_buffers(item).unwrap();
        pass.set_index_buffer(buffers.index_buffer.slice(..), 0, IndexFormat::Uint32);
        pass.draw_indexed(0..buffers.index_count, 0, 0..1);
        RenderCommandResult::Success
    }
}
