use super::extract::CuboidsBuffer;
use crate::component::*;

use bevy::prelude::*;
use bevy::render::{render_resource::*, renderer::RenderDevice};

/// Runs in `CoreStage::PostUpdate` to create GPU buffers early. We want extraction to be as simple as cloning a buffer handle.
pub(crate) fn create_new_instance_buffers(
    mut commands: Commands,
    device: Res<RenderDevice>,
    query: Query<
        (Entity, &Cuboids, Option<&CuboidsMask>),
        Or<(
            Added<Cuboids>,
            Changed<Cuboids>,
            Added<CuboidsMask>,
            Changed<CuboidsMask>,
        )>,
    >,
) {
    for (entity, instances, maybe_mask) in query.iter() {
        let mut visible_instances = Vec::new();
        let render_instances: &[Cuboid] = if let Some(mask) = maybe_mask {
            assert_eq!(mask.bitmask.len(), instances.instances.len());
            visible_instances.reserve(mask.bitmask.count_zeros());
            for index in mask.bitmask.iter_zeros() {
                visible_instances.push(instances.instances[index]);
            }
            &visible_instances
        } else {
            &instances.instances
        };
        let buffer = device.create_buffer_with_data(&BufferInitDescriptor {
            label: Some("instance data buffer"),
            contents: bytemuck::cast_slice(render_instances),
            usage: BufferUsages::VERTEX | BufferUsages::COPY_DST,
        });
        commands.get_or_spawn(entity).insert(CuboidsBuffer {
            buffer,
            length: render_instances.len(),
        });
    }
}
