use super::bundle::CubeMesh;
use super::draw_function::DrawCuboids;
use super::extract::CuboidsBuffer;
use super::prepare::create_new_instance_buffers;
use super::pipeline::{SHADER_HANDLE, queue_instances, CuboidInstancingPipeline};

use bevy::{
    core_pipeline::Transparent3d,
    prelude::*,
    render::{
        render_component::ExtractComponentPlugin, render_phase::AddRenderCommand,
        render_resource::*, RenderApp, RenderStage,
    },
};

/// Renders the [`CuboidInstancesBundle`](crate::CuboidInstancesBundle) components using the canonical mesh instancing
/// technique.
pub struct InstancingRenderPlugin;

impl Plugin for InstancingRenderPlugin {
    fn build(&self, app: &mut App) {
        let mut meshes = app.world.get_resource_mut::<Assets<Mesh>>().unwrap();
        let cube_mesh = CubeMesh::new(&mut meshes);

        let mut shaders = app.world.get_resource_mut::<Assets<Shader>>().unwrap();
        let wgsl_src = include_str!("instancing.wgsl");
        shaders.set_untracked(SHADER_HANDLE, Shader::from_wgsl(wgsl_src));

        app.insert_resource(cube_mesh)
            .add_plugin(ExtractComponentPlugin::<CuboidsBuffer>::default())
            .add_system_to_stage(CoreStage::PostUpdate, create_new_instance_buffers);

        app.sub_app_mut(RenderApp)
            .add_render_command::<Transparent3d, DrawCuboids>()
            .init_resource::<CuboidInstancingPipeline>()
            .init_resource::<SpecializedMeshPipelines<CuboidInstancingPipeline>>()
            .add_system_to_stage(RenderStage::Queue, queue_instances);
    }
}
