use crate::Cuboids;

use bevy::prelude::*;

/// The components needed to render [`Cuboids`] with the [`InstancingRenderPlugin`](crate::InstancingRenderPlugin).
#[derive(Bundle)]
pub struct CuboidInstancesBundle {
    /// A mesh created from [`CubeMesh`].
    pub cube_mesh: Handle<Mesh>,
    /// The instances of `cube_mesh` to be rendered.
    pub instances: Cuboids,
    pub transform: Transform,
    pub global_transform: GlobalTransform,
    pub computed_visibility: ComputedVisibility,
    pub visibility: Visibility,
}

impl CuboidInstancesBundle {
    /// Use the `Res<CubeMesh>` resource created by [`InstancingRenderPlugin`](crate::InstancingRenderPlugin) to access a
    /// compatible `cube_mesh` handle.
    pub fn new(instances: Cuboids, cube_mesh: Handle<Mesh>) -> Self {
        Self {
            instances,
            cube_mesh,
            transform: Default::default(),
            global_transform: Default::default(),
            computed_visibility: ComputedVisibility::default(),
            visibility: Visibility::default(),
        }
    }
}

/// A cube mesh that can be acccessed through `Res<CubeMesh>` while creating a [`CuboidInstancesBundle`].
#[derive(Clone, Debug)]
pub struct CubeMesh {
    handle: Handle<Mesh>,
}

impl CubeMesh {
    pub fn new(meshes: &mut Assets<Mesh>) -> Self {
        let cube = shape::Box {
            min_x: 0.0,
            max_x: 1.0,
            min_y: 0.0,
            max_y: 1.0,
            min_z: 0.0,
            max_z: 1.0,
        };
        Self {
            handle: meshes.add(Mesh::from(cube)),
        }
    }

    pub fn handle(&self) -> &Handle<Mesh> {
        &self.handle
    }
}
