use crate::bundle::CubeMesh;
use crate::component::{create_new_instance_buffers, CuboidInstancesBuffer};
use crate::draw_function::DrawCuboidInstances;
use crate::pipeline::SHADER_HANDLE;
use crate::pipeline::{queue_instances, CuboidInstancingPipeline};

use bevy::{
    core_pipeline::Transparent3d,
    prelude::*,
    render::{
        render_component::ExtractComponentPlugin, render_phase::AddRenderCommand,
        render_resource::*, RenderApp, RenderStage,
    },
};

/// Creates a new pipeline in Bevy's render graph to support instanced rendering of all entities with a
/// [`CuboidInstancesBundle`](crate::CuboidInstancesBundle) of components.
pub struct InstancingRenderPlugin;

impl Plugin for InstancingRenderPlugin {
    fn build(&self, app: &mut App) {
        let mut meshes = app.world.get_resource_mut::<Assets<Mesh>>().unwrap();
        let cube_mesh = CubeMesh::new(&mut meshes);

        let mut shaders = app.world.get_resource_mut::<Assets<Shader>>().unwrap();
        let wgsl_src = include_str!("../assets/shaders/instancing.wgsl");
        shaders.set_untracked(SHADER_HANDLE, Shader::from_wgsl(wgsl_src));

        app.insert_resource(cube_mesh)
            .add_plugin(ExtractComponentPlugin::<CuboidInstancesBuffer>::default())
            .add_system_to_stage(CoreStage::PostUpdate, create_new_instance_buffers);

        app.sub_app_mut(RenderApp)
            .add_render_command::<Transparent3d, DrawCuboidInstances>()
            .init_resource::<CuboidInstancingPipeline>()
            .init_resource::<SpecializedMeshPipelines<CuboidInstancingPipeline>>()
            .add_system_to_stage(RenderStage::Queue, queue_instances);
    }
}
