use lazy_static::lazy_static;
use std::env;
use std::io;
use std::iter;
use std::path::{Component, Path, PathBuf};
use walkdir::WalkDir;

lazy_static! {
    static ref CURR_DIR: PathBuf = env::current_dir().unwrap();
}

#[derive(Debug)]
pub(crate) struct FilesystemTree {
    paths: Vec<PathBuf>,
}

#[derive(Debug)]
pub(crate) struct FilesystemEntry {
    path: PathBuf, // absolute path
    kind: EntryKind,
}

#[derive(Debug)]
pub(crate) enum EntryKind {
    File,
    Dir,
    Symlink,
}

#[derive(Debug)]
pub(crate) struct Error {
    kind: ErrorKind,
    path: Option<PathBuf>,
}

#[derive(Debug)]
pub(crate) enum ErrorKind {
    PermissionDenied,
    NotFound,
}

impl Error {
    pub(crate) fn path(&self) -> Option<&Path> {
        match &self.path {
            Some(p) => Some(p.as_path()),
            None => None,
        }
    }

    pub(crate) fn kind(&self) -> &ErrorKind {
        &self.kind
    }
}

impl From<io::ErrorKind> for ErrorKind {
    fn from(kind: io::ErrorKind) -> Self {
        match kind {
            io::ErrorKind::PermissionDenied => ErrorKind::PermissionDenied,
            io::ErrorKind::NotFound => ErrorKind::NotFound,
            _ => unimplemented!(),
        }
    }
}

impl<P: AsRef<Path>> From<&[P]> for FilesystemTree {
    fn from(paths: &[P]) -> Self {
        let paths = reduce_paths(paths);
        FilesystemTree { paths }
    }
}

impl FilesystemTree {
    pub(crate) fn iter(&self) -> impl Iterator<Item = Result<FilesystemEntry, Error>> + '_ {
        self.paths.iter().flat_map(|p| {
            let iter_a;
            let iter_b;

            match p.symlink_metadata() {
                Ok(metadata) => {
                    if metadata.file_type().is_file() {
                        iter_a = Some(iter::once(Ok(FilesystemEntry {
                            path: p.clone(),
                            kind: EntryKind::File,
                        })));
                        iter_b = None;
                    } else if metadata.file_type().is_dir() {
                        iter_a = None;
                        iter_b =
                            Some(WalkDir::new(p).sort_by_file_name().into_iter().map(
                                |e| match e {
                                    Ok(e) => Ok(FilesystemEntry {
                                        path: e.into_path(),
                                        kind: EntryKind::Dir,
                                    }),
                                    Err(err) => Err(Error {
                                        path: match err.path() {
                                            Some(p) => Some(PathBuf::from(p)),
                                            None => None,
                                        },
                                        kind: err.into_io_error().unwrap().kind().into(),
                                    }),
                                },
                            ));
                    } else if metadata.file_type().is_symlink() {
                        iter_a = Some(iter::once(Ok(FilesystemEntry {
                            path: p.clone(),
                            kind: EntryKind::Symlink,
                        })));
                        iter_b = None;
                    } else {
                        unimplemented!("Unknown file type.");
                    }
                }
                Err(err) => {
                    iter_a = Some(iter::once(Err(Error {
                        path: Some(p.clone()),
                        kind: err.kind().into(),
                    })));
                    iter_b = None;
                }
            }

            return iter_a
                .into_iter()
                .flatten()
                .chain(iter_b.into_iter().flatten());
        })
    }
}

fn reduce_paths<P: AsRef<Path>>(paths: &[P]) -> Vec<PathBuf> {
    // Make all paths absolute
    let mut paths: Vec<PathBuf> = paths
        .into_iter()
        .map(|p| {
            let mut components = p.as_ref().components().peekable();
            let mut ret = if let Some(c @ Component::Prefix(..)) = components.peek().cloned() {
                components.next();
                PathBuf::from(c.as_os_str())
            } else {
                CURR_DIR.clone()
            };

            for component in components {
                match component {
                    Component::Normal(c) => {
                        ret.push(c);
                    }
                    Component::RootDir => {
                        ret.push(component.as_os_str());
                    }
                    Component::ParentDir => {
                        ret.pop();
                    }
                    Component::CurDir => {}
                    Component::Prefix(_) => unreachable!(),
                }
            }

            ret
        })
        .collect();

    // Sort paths lexicographically
    paths.sort_unstable();

    // Remove paths which are children of other paths
    if paths.len() >= 2 {
        let mut skip = true;
        let mut f_path = paths[0].clone();
        paths.retain(|p| {
            // Skip the first path
            if skip {
                skip = false;
                return true;
            // Remove subpaths of f_path
            } else if p.starts_with(&f_path) {
                return false;
            // Keep
            } else {
                f_path = p.clone();
                return true;
            }
        });
    }

    return paths;
}
