// Copyright (c) Facebook, Inc. and its affiliates.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

use crate::{CommonField, DumpField};
use model::EnumIter;
use model::{
    FieldId, NetworkModelFieldId, SingleCgroupModelFieldId, SingleDiskModelFieldId,
    SingleNetModelFieldId, SingleProcessModelFieldId, SystemModelFieldId,
};

use anyhow::{bail, Error, Result};
use once_cell::sync::Lazy;
use regex::Regex;
use std::str::FromStr;
use std::string::ToString;
use structopt::StructOpt;

/// Field that represents a group of related FieldIds of a Queriable.
/// Shorthand for specifying fields to dump.
pub trait AggField<F: FieldId> {
    fn expand(&self, detail: bool) -> Vec<F>;
}

/// Generic representation of fields accepted by different dump subcommands.
/// Each DumpOptionField is either an aggregation of multiple FieldIds, or a
/// "unit" field which could be either a CommonField or a FieldId.
#[derive(Clone, Debug, PartialEq)]
pub enum DumpOptionField<F: FieldId, A: AggField<F>> {
    Unit(DumpField<F>),
    Agg(A),
}

/// Expand the Agg fields and collect them with other Unit fields.
pub fn expand_fields<F: FieldId + Clone, A: AggField<F>>(
    fields: &[DumpOptionField<F, A>],
    detail: bool,
) -> Vec<DumpField<F>> {
    let mut res = Vec::new();
    for field in fields {
        match field {
            DumpOptionField::Unit(field) => res.push(field.clone()),
            DumpOptionField::Agg(agg) => {
                res.extend(agg.expand(detail).into_iter().map(DumpField::FieldId))
            }
        }
    }
    res
}

/// Used by structopt to parse user provided --fields.
impl<F: FieldId + FromStr, A: AggField<F> + FromStr> FromStr for DumpOptionField<F, A> {
    type Err = Error;

    /// When parsing command line options into DumpOptionField, priority order
    /// is CommonField, AggField, and then FieldId.
    fn from_str(s: &str) -> Result<Self, Self::Err> {
        if let Ok(common) = CommonField::from_str(s) {
            Ok(Self::Unit(DumpField::Common(common)))
        } else if let Ok(agg) = A::from_str(s) {
            Ok(Self::Agg(agg))
        } else if let Ok(field_id) = F::from_str(s) {
            Ok(Self::Unit(DumpField::FieldId(field_id)))
        } else {
            bail!("Variant not found: {}", s);
        }
    }
}

/// Used for generating help string that lists all supported fields.
impl<F: FieldId + ToString, A: AggField<F> + ToString> ToString for DumpOptionField<F, A> {
    fn to_string(&self) -> String {
        match self {
            Self::Unit(DumpField::Common(common)) => common.to_string(),
            Self::Unit(DumpField::FieldId(field_id)) => field_id.to_string(),
            Self::Agg(agg) => agg.to_string(),
        }
    }
}

/// Join stringified items with ", ". Used for generating help string that lists
/// all supported fields.
fn join(iter: impl IntoIterator<Item = impl ToString>) -> String {
    iter.into_iter()
        .map(|v| v.to_string())
        .collect::<Vec<_>>()
        .join(", ")
}

// make_option macro will build a enum of tags that map to string values by
// implementing the FromStr trait.
// This is useful when are trying to processing or display fields base on
// a user's input. Here's a use case:
// We display fields in the order of user's input. After we got
// the input array, dfill trait will automatically generate a vec of fns base
// on that array. For example, user input `--fields cpu_usage cpu_user`,
// enum generated by make_option will auto translate string to enum tags. After
// that dfill trait will generate `vec![print_cpu_usage, print_cpu_user]`. And
// the dprint trait will just iterate over the fns and call it with current model.
//
// Another user case is for the select feature, we don't want a giant match
// of string patterns once user select some field to do some operations. Instead,
// we can use a match of enum tags, that will be much faster.
macro_rules! make_option {
    ($name:ident {$($str_field:tt: $enum_field:ident,)*}) => {
        #[derive(Debug, Clone, Copy, PartialEq, Hash, Eq)]
        pub enum $name {
            $($enum_field,)*
        }

        impl FromStr for $name {
            type Err = Error;

            fn from_str(opt: &str) -> Result<Self> {
                match opt.to_lowercase().as_str() {
                    $($str_field => Ok($name::$enum_field),)*
                    _ => bail!("Fail to parse {}", opt)
                }
            }
        }
    }
}

/// Represents the four sub-model of SystemModel.
#[derive(
    Clone,
    Debug,
    PartialEq,
    below_derive::EnumFromStr,
    below_derive::EnumToString
)]
pub enum SystemAggField {
    Cpu,
    Mem,
    Vm,
    Stat,
}

impl AggField<SystemModelFieldId> for SystemAggField {
    fn expand(&self, detail: bool) -> Vec<SystemModelFieldId> {
        use model::MemoryModelFieldId as Mem;
        use model::ProcStatModelFieldId as Stat;
        use model::SingleCpuModelFieldId as Cpu;
        use model::SystemModelFieldId as FieldId;
        use model::VmModelFieldId as Vm;

        if detail {
            match self {
                Self::Cpu => Cpu::unit_variant_iter()
                    // The Idx field is always -1 (we aggregate all CPUs)
                    .filter(|v| v != &Cpu::Idx)
                    .map(FieldId::Cpu)
                    .collect(),
                Self::Mem => Mem::unit_variant_iter().map(FieldId::Mem).collect(),
                Self::Vm => Vm::unit_variant_iter().map(FieldId::Vm).collect(),
                Self::Stat => Stat::unit_variant_iter().map(FieldId::Stat).collect(),
            }
        } else {
            // Default fields for each group
            match self {
                Self::Cpu => vec![Cpu::UsagePct, Cpu::UserPct, Cpu::SystemPct]
                    .into_iter()
                    .map(FieldId::Cpu)
                    .collect(),
                Self::Mem => vec![Mem::Total, Mem::Free]
                    .into_iter()
                    .map(FieldId::Mem)
                    .collect(),
                Self::Vm => Vm::unit_variant_iter().map(FieldId::Vm).collect(),
                Self::Stat => Stat::unit_variant_iter().map(FieldId::Stat).collect(),
            }
        }
    }
}

pub type SystemOptionField = DumpOptionField<SystemModelFieldId, SystemAggField>;

pub static DEFAULT_SYSTEM_FIELDS: &[SystemOptionField] = &[
    DumpOptionField::Unit(DumpField::FieldId(SystemModelFieldId::Hostname)),
    DumpOptionField::Unit(DumpField::Common(CommonField::Datetime)),
    DumpOptionField::Agg(SystemAggField::Cpu),
    DumpOptionField::Agg(SystemAggField::Mem),
    DumpOptionField::Agg(SystemAggField::Vm),
    DumpOptionField::Unit(DumpField::FieldId(SystemModelFieldId::KernelVersion)),
    DumpOptionField::Unit(DumpField::FieldId(SystemModelFieldId::OsRelease)),
    DumpOptionField::Agg(SystemAggField::Stat),
    DumpOptionField::Unit(DumpField::Common(CommonField::Timestamp)),
];

const SYSTEM_ABOUT: &str = "Dump system stats";

/// Generated about message for System dump so supported fields are up-to-date.
static SYSTEM_LONG_ABOUT: Lazy<String> = Lazy::new(|| {
    format!(
        r#"{about}

********************** Available fields **********************

{common_fields}, {system_fields}

{all_cpu_fields}

cpus.N.<cpu_field> for individual CPU data. N is individual CPU index and
<cpu_field> is same as cpu fields above with `cpu.` prefix stripped.

{all_memory_fields}

{all_vm_fields}

{all_stat_fields}

********************** Aggregated fields **********************

* cpu: includes [{agg_cpu_fields}].

* mem: includes [{agg_memory_fields}].

* vm: includes [{agg_vm_fields}].

* stat: includes [{agg_stat_fields}].

* --detail: includes [<agg_field>.*] for each given aggregated field.

* --default: includes [{default_fields}].

* --everything: includes everything (equivalent to --default --detail).

********************** Example Commands **********************

$ below dump system -b "08:30:00" -e "08:30:30" -f datetime vm hostname -O csv

"#,
        about = SYSTEM_ABOUT,
        common_fields = join(CommonField::unit_variant_iter()),
        system_fields = join(SystemModelFieldId::unit_variant_iter()),
        all_cpu_fields = join(SystemAggField::Cpu.expand(true)),
        all_memory_fields = join(SystemAggField::Mem.expand(true)),
        all_vm_fields = join(SystemAggField::Vm.expand(true)),
        all_stat_fields = join(SystemAggField::Stat.expand(true)),
        agg_cpu_fields = join(SystemAggField::Cpu.expand(false)),
        agg_memory_fields = join(SystemAggField::Mem.expand(false)),
        agg_vm_fields = join(SystemAggField::Vm.expand(false)),
        agg_stat_fields = join(SystemAggField::Stat.expand(false)),
        default_fields = join(DEFAULT_SYSTEM_FIELDS.to_owned()),
    )
});

#[derive(
    Clone,
    Debug,
    PartialEq,
    below_derive::EnumFromStr,
    below_derive::EnumToString
)]
pub enum DiskAggField {
    Read,
    Write,
    Discard,
}

impl AggField<SingleDiskModelFieldId> for DiskAggField {
    fn expand(&self, _detail: bool) -> Vec<SingleDiskModelFieldId> {
        use model::SingleDiskModelFieldId::*;

        match self {
            Self::Read => vec![
                ReadBytesPerSec,
                ReadCompleted,
                ReadMerged,
                ReadSectors,
                TimeSpendReadMs,
            ],
            Self::Write => vec![
                WriteBytesPerSec,
                WriteCompleted,
                WriteMerged,
                WriteSectors,
                TimeSpendWriteMs,
            ],
            Self::Discard => vec![
                DiscardBytesPerSec,
                DiscardCompleted,
                DiscardMerged,
                DiscardSectors,
                TimeSpendDiscardMs,
            ],
        }
    }
}

pub type DiskOptionField = DumpOptionField<SingleDiskModelFieldId, DiskAggField>;

pub static DEFAULT_DISK_FIELDS: &[DiskOptionField] = &[
    DumpOptionField::Unit(DumpField::Common(CommonField::Datetime)),
    DumpOptionField::Unit(DumpField::FieldId(SingleDiskModelFieldId::Name)),
    DumpOptionField::Unit(DumpField::FieldId(
        SingleDiskModelFieldId::DiskTotalBytesPerSec,
    )),
    DumpOptionField::Unit(DumpField::FieldId(SingleDiskModelFieldId::Major)),
    DumpOptionField::Unit(DumpField::FieldId(SingleDiskModelFieldId::Minor)),
    DumpOptionField::Agg(DiskAggField::Read),
    DumpOptionField::Agg(DiskAggField::Write),
    DumpOptionField::Agg(DiskAggField::Discard),
    DumpOptionField::Unit(DumpField::Common(CommonField::Timestamp)),
];

const DISK_ABOUT: &str = "Dump disk stats";

/// Generated about message for System dump so supported fields are up-to-date.
static DISK_LONG_ABOUT: Lazy<String> = Lazy::new(|| {
    format!(
        r#"{about}

********************** Available fields **********************

{common_fields}, and expanded fields below.

********************** Aggregated fields **********************

* read: includes [{agg_read_fields}].

* write: includes [{agg_write_fields}].

* discard: includes [{agg_discard_fields}].

* --detail: no effect.

* --default: includes [{default_fields}].

* --everything: includes everything (equivalent to --default --detail).

********************** Example Commands **********************

Simple example:

$ below dump disk -b "08:30:00" -e "08:30:30" -f read write discard -O csv

Output stats for all "nvme0*" matched disk from 08:30:00 to 08:30:30:

$ below dump disk -b "08:30:00" -e "08:30:30" -s name -F nvme0* -O json

Output stats for top 5 read partitions for each time slice from 08:30:00 to 08:30:30:

$ below dump disk -b "08:30:00" -e "08:30:30" -s read_bytes_per_sec --rsort --top 5

"#,
        about = DISK_ABOUT,
        common_fields = join(CommonField::unit_variant_iter()),
        agg_read_fields = join(DiskAggField::Read.expand(false)),
        agg_write_fields = join(DiskAggField::Write.expand(false)),
        agg_discard_fields = join(DiskAggField::Discard.expand(false)),
        default_fields = join(DEFAULT_DISK_FIELDS.to_owned()),
    )
});

/// Represents the four sub-model of ProcessModel.
#[derive(
    Clone,
    Debug,
    PartialEq,
    below_derive::EnumFromStr,
    below_derive::EnumToString
)]
pub enum ProcessAggField {
    Cpu,
    Mem,
    Io,
}

impl AggField<SingleProcessModelFieldId> for ProcessAggField {
    fn expand(&self, detail: bool) -> Vec<SingleProcessModelFieldId> {
        use model::ProcessCpuModelFieldId as Cpu;
        use model::ProcessIoModelFieldId as Io;
        use model::ProcessMemoryModelFieldId as Mem;
        use model::SingleProcessModelFieldId as FieldId;

        if detail {
            match self {
                Self::Cpu => Cpu::unit_variant_iter().map(FieldId::Cpu).collect(),
                Self::Mem => Mem::unit_variant_iter().map(FieldId::Mem).collect(),
                Self::Io => Io::unit_variant_iter().map(FieldId::Io).collect(),
            }
        } else {
            // Default fields for each group
            match self {
                Self::Cpu => vec![FieldId::Cpu(Cpu::UsagePct)],
                Self::Mem => vec![FieldId::Mem(Mem::RssBytes)],
                Self::Io => vec![FieldId::Io(Io::RbytesPerSec), FieldId::Io(Io::WbytesPerSec)],
            }
        }
    }
}

pub type ProcessOptionField = DumpOptionField<SingleProcessModelFieldId, ProcessAggField>;

pub static DEFAULT_PROCESS_FIELDS: &[ProcessOptionField] = &[
    DumpOptionField::Unit(DumpField::Common(CommonField::Datetime)),
    DumpOptionField::Unit(DumpField::FieldId(SingleProcessModelFieldId::Pid)),
    DumpOptionField::Unit(DumpField::FieldId(SingleProcessModelFieldId::Ppid)),
    DumpOptionField::Unit(DumpField::FieldId(SingleProcessModelFieldId::Comm)),
    DumpOptionField::Unit(DumpField::FieldId(SingleProcessModelFieldId::State)),
    DumpOptionField::Agg(ProcessAggField::Cpu),
    DumpOptionField::Agg(ProcessAggField::Mem),
    DumpOptionField::Agg(ProcessAggField::Io),
    DumpOptionField::Unit(DumpField::FieldId(SingleProcessModelFieldId::UptimeSecs)),
    DumpOptionField::Unit(DumpField::FieldId(SingleProcessModelFieldId::Cgroup)),
    DumpOptionField::Unit(DumpField::Common(CommonField::Timestamp)),
    DumpOptionField::Unit(DumpField::FieldId(SingleProcessModelFieldId::Cmdline)),
    DumpOptionField::Unit(DumpField::FieldId(SingleProcessModelFieldId::ExePath)),
];

const PROCESS_ABOUT: &str = "Dump process stats";

/// Generated about message for Process dump so supported fields are up-to-date.
static PROCESS_LONG_ABOUT: Lazy<String> = Lazy::new(|| {
    format!(
        r#"{about}

********************** Available fields **********************

{common_fields}, {process_fields}

{all_cpu_fields}

{all_memory_fields}

{all_io_fields}

********************** Aggregated fields **********************

* cpu: includes [{agg_cpu_fields}].

* mem: includes [{agg_memory_fields}].

* io: includes [{agg_io_fields}].

* --detail: includes [<agg_field>.*] for each given aggregated field.

* --default: includes [{default_fields}].

* --everything: includes everything (equivalent to --default --detail).

********************** Example Commands **********************

Simple example:

$ below dump process -b "08:30:00" -e "08:30:30" -f comm cpu io.rwbytes_per_sec -O csv

Output stats for all "below*" matched processes from 08:30:00 to 08:30:30:

$ below dump process -b "08:30:00" -e "08:30:30" -s comm -F below* -O json

Output stats for top 5 CPU intense processes for each time slice from 08:30:00 to 08:30:30:

$ below dump process -b "08:30:00" -e "08:30:30" -s cpu.usage_pct --rsort --top 5

"#,
        about = PROCESS_ABOUT,
        common_fields = join(CommonField::unit_variant_iter()),
        process_fields = join(SingleProcessModelFieldId::unit_variant_iter()),
        all_cpu_fields = join(ProcessAggField::Cpu.expand(true)),
        all_memory_fields = join(ProcessAggField::Mem.expand(true)),
        all_io_fields = join(ProcessAggField::Io.expand(true)),
        agg_cpu_fields = join(ProcessAggField::Cpu.expand(false)),
        agg_memory_fields = join(ProcessAggField::Mem.expand(false)),
        agg_io_fields = join(ProcessAggField::Io.expand(false)),
        default_fields = join(DEFAULT_PROCESS_FIELDS.to_owned()),
    )
});

/// Represents the four sub-model of SingleCgroupModel.
#[derive(
    Clone,
    Debug,
    PartialEq,
    below_derive::EnumFromStr,
    below_derive::EnumToString
)]
pub enum CgroupAggField {
    Cpu,
    Mem,
    Io,
    Pressure,
}

impl AggField<SingleCgroupModelFieldId> for CgroupAggField {
    fn expand(&self, detail: bool) -> Vec<SingleCgroupModelFieldId> {
        use model::CgroupCpuModelFieldId as Cpu;
        use model::CgroupIoModelFieldId as Io;
        use model::CgroupMemoryModelFieldId as Mem;
        use model::CgroupPressureModelFieldId as Pressure;
        use model::SingleCgroupModelFieldId as FieldId;

        if detail {
            match self {
                Self::Cpu => Cpu::unit_variant_iter().map(FieldId::Cpu).collect(),
                Self::Mem => Mem::unit_variant_iter().map(FieldId::Mem).collect(),
                Self::Io => Io::unit_variant_iter().map(FieldId::Io).collect(),
                Self::Pressure => Pressure::unit_variant_iter()
                    .map(FieldId::Pressure)
                    .collect(),
            }
        } else {
            // Default fields for each group
            match self {
                Self::Cpu => vec![FieldId::Cpu(Cpu::UsagePct)],
                Self::Mem => vec![FieldId::Mem(Mem::Total)],
                Self::Io => vec![FieldId::Io(Io::RbytesPerSec), FieldId::Io(Io::WbytesPerSec)],
                Self::Pressure => vec![
                    FieldId::Pressure(Pressure::CpuSomePct),
                    FieldId::Pressure(Pressure::MemoryFullPct),
                    FieldId::Pressure(Pressure::IoFullPct),
                ],
            }
        }
    }
}

pub type CgroupOptionField = DumpOptionField<SingleCgroupModelFieldId, CgroupAggField>;

pub static DEFAULT_CGROUP_FIELDS: &[CgroupOptionField] = &[
    DumpOptionField::Unit(DumpField::FieldId(SingleCgroupModelFieldId::Name)),
    DumpOptionField::Unit(DumpField::FieldId(SingleCgroupModelFieldId::InodeNumber)),
    DumpOptionField::Unit(DumpField::Common(CommonField::Datetime)),
    DumpOptionField::Agg(CgroupAggField::Cpu),
    DumpOptionField::Agg(CgroupAggField::Mem),
    DumpOptionField::Agg(CgroupAggField::Io),
    DumpOptionField::Agg(CgroupAggField::Pressure),
    DumpOptionField::Unit(DumpField::Common(CommonField::Timestamp)),
];

const CGROUP_ABOUT: &str = "Dump cgroup stats";

/// Generated about message for Cgroup dump so supported fields are up-to-date.
static CGROUP_LONG_ABOUT: Lazy<String> = Lazy::new(|| {
    format!(
        r#"{about}

********************** Available fields **********************

{common_fields}, {cgroup_fields}

{all_cpu_fields}

{all_memory_fields}

{all_io_fields}

{all_pressure_fields}

********************** Aggregated fields **********************

* cpu: includes [{agg_cpu_fields}].

* mem: includes [{agg_memory_fields}].

* io: includes [{agg_io_fields}].

* pressure: includes [{agg_pressure_fields}].

* --detail: includes [<agg_field>.*] for each given aggregated field.

* --default: includes [{default_fields}].

* --everything: includes everything (equivalent to --default --detail).

********************** Example Commands **********************

Simple example:

$ below dump cgroup -b "08:30:00" -e "08:30:30" -f name cpu -O csv

Output stats for all cgroups matching pattern "below*" for time slices
from 08:30:00 to 08:30:30:

$ below dump cgroup -b "08:30:00" -e "08:30:30" -s name -F below* -O json

Output stats for top 5 CPU intense cgroups for each time slice
from 08:30:00 to 08:30:30 recursively:

$ below dump cgroup -b "08:30:00" -e "08:30:30" -s cpu.usage_pct --rsort --top 5

"#,
        about = CGROUP_ABOUT,
        common_fields = join(CommonField::unit_variant_iter()),
        cgroup_fields = join(SingleCgroupModelFieldId::unit_variant_iter()),
        all_cpu_fields = join(CgroupAggField::Cpu.expand(true)),
        all_memory_fields = join(CgroupAggField::Mem.expand(true)),
        all_io_fields = join(CgroupAggField::Io.expand(true)),
        all_pressure_fields = join(CgroupAggField::Pressure.expand(true)),
        agg_cpu_fields = join(CgroupAggField::Cpu.expand(false)),
        agg_memory_fields = join(CgroupAggField::Mem.expand(false)),
        agg_io_fields = join(CgroupAggField::Io.expand(false)),
        agg_pressure_fields = join(CgroupAggField::Pressure.expand(false)),
        default_fields = join(DEFAULT_CGROUP_FIELDS.to_owned()),
    )
});

/// Represents the iface sub-models of network model.
#[derive(
    Clone,
    Debug,
    PartialEq,
    below_derive::EnumFromStr,
    below_derive::EnumToString
)]
pub enum IfaceAggField {
    Rate,
    Rx,
    Tx,
}

impl AggField<SingleNetModelFieldId> for IfaceAggField {
    fn expand(&self, _detail: bool) -> Vec<SingleNetModelFieldId> {
        use model::SingleNetModelFieldId::*;
        match self {
            Self::Rate => vec![
                RxBytesPerSec,
                TxBytesPerSec,
                ThroughputPerSec,
                RxPacketsPerSec,
                TxPacketsPerSec,
            ],
            Self::Rx => vec![
                RxBytes,
                RxCompressed,
                RxCrcErrors,
                RxDropped,
                RxErrors,
                RxFifoErrors,
                RxFrameErrors,
                RxLengthErrors,
                RxMissedErrors,
                RxNohandler,
                RxOverErrors,
                RxPackets,
            ],
            Self::Tx => vec![
                TxAbortedErrors,
                TxBytes,
                TxCarrierErrors,
                TxCompressed,
                TxDropped,
                TxErrors,
                TxFifoErrors,
                TxHeartbeatErrors,
                TxPackets,
                TxWindowErrors,
            ],
        }
    }
}

pub type IfaceOptionField = DumpOptionField<SingleNetModelFieldId, IfaceAggField>;

pub static DEFAULT_IFACE_FIELDS: &[IfaceOptionField] = &[
    DumpOptionField::Unit(DumpField::Common(CommonField::Datetime)),
    DumpOptionField::Unit(DumpField::FieldId(SingleNetModelFieldId::Collisions)),
    DumpOptionField::Unit(DumpField::FieldId(SingleNetModelFieldId::Multicast)),
    DumpOptionField::Unit(DumpField::FieldId(SingleNetModelFieldId::Interface)),
    DumpOptionField::Agg(IfaceAggField::Rate),
    DumpOptionField::Agg(IfaceAggField::Rx),
    DumpOptionField::Agg(IfaceAggField::Tx),
    DumpOptionField::Unit(DumpField::Common(CommonField::Timestamp)),
];

const IFACE_ABOUT: &str = "Dump the link layer iface stats";

/// Generated about message for Iface dump so supported fields are up-to-date.
static IFACE_LONG_ABOUT: Lazy<String> = Lazy::new(|| {
    format!(
        r#"{about}

********************** Available fields **********************

{common_fields}, and expanded fields below.

********************** Aggregated fields **********************

* rate: includes [{agg_rate_fields}].

* rx: includes [{agg_rx_fields}].

* tx: includes [{agg_tx_fields}].

* --detail: no effect.

* --default: includes [{default_fields}].

* --everything: includes everything (equivalent to --default --detail).

********************** Example Commands **********************

Simple example:

$ below dump iface -b "08:30:00" -e "08:30:30" -f interface rate -O csv

Output stats for all iface stats matching pattern "eth*" for time slices
from 08:30:00 to 08:30:30:

$ below dump iface -b "08:30:00" -e "08:30:30" -s interface -F eth* -O json

"#,
        about = IFACE_ABOUT,
        common_fields = join(CommonField::unit_variant_iter()),
        agg_rate_fields = join(IfaceAggField::Rate.expand(false)),
        agg_rx_fields = join(IfaceAggField::Rx.expand(false)),
        agg_tx_fields = join(IfaceAggField::Tx.expand(false)),
        default_fields = join(DEFAULT_IFACE_FIELDS.to_owned()),
    )
});

/// Represents the ip and icmp sub-models of the network model.
#[derive(
    Clone,
    Debug,
    PartialEq,
    below_derive::EnumFromStr,
    below_derive::EnumToString
)]
pub enum NetworkAggField {
    Ip,
    Ip6,
    Icmp,
    Icmp6,
}

impl AggField<NetworkModelFieldId> for NetworkAggField {
    fn expand(&self, _detail: bool) -> Vec<NetworkModelFieldId> {
        use model::NetworkModelFieldId as FieldId;
        match self {
            Self::Ip => model::IpModelFieldId::unit_variant_iter()
                .map(FieldId::Ip)
                .collect(),
            Self::Ip6 => model::Ip6ModelFieldId::unit_variant_iter()
                .map(FieldId::Ip6)
                .collect(),
            Self::Icmp => model::IcmpModelFieldId::unit_variant_iter()
                .map(FieldId::Icmp)
                .collect(),
            Self::Icmp6 => model::Icmp6ModelFieldId::unit_variant_iter()
                .map(FieldId::Icmp6)
                .collect(),
        }
    }
}

pub type NetworkOptionField = DumpOptionField<NetworkModelFieldId, NetworkAggField>;

pub static DEFAULT_NETWORK_FIELDS: &[NetworkOptionField] = &[
    DumpOptionField::Unit(DumpField::Common(CommonField::Datetime)),
    DumpOptionField::Agg(NetworkAggField::Ip),
    DumpOptionField::Agg(NetworkAggField::Ip6),
    DumpOptionField::Agg(NetworkAggField::Icmp),
    DumpOptionField::Agg(NetworkAggField::Icmp6),
    DumpOptionField::Unit(DumpField::Common(CommonField::Timestamp)),
];

const NETWORK_ABOUT: &str = "Dump the network layer stats including ip and icmp";

/// Generated about message for Network dump so supported fields are up-to-date.
static NETWORK_LONG_ABOUT: Lazy<String> = Lazy::new(|| {
    format!(
        r#"{about}

********************** Available fields **********************

{common_fields}, and expanded fields below.

********************** Aggregated fields **********************

* ip: includes [{agg_ip_fields}].

* ip6: includes [{agg_ip6_fields}].

* icmp: includes [{agg_icmp_fields}].

* icmp6: includes [{agg_icmp6_fields}].

* --detail: no effect.

* --default: includes [{default_fields}].

* --everything: includes everything (equivalent to --default --detail).

********************** Example Commands **********************

Example:

$ below dump network -b "08:30:00" -e "08:30:30" -f ip ip6 -O json

"#,
        about = NETWORK_ABOUT,
        common_fields = join(CommonField::unit_variant_iter()),
        agg_ip_fields = join(NetworkAggField::Ip.expand(false)),
        agg_ip6_fields = join(NetworkAggField::Ip6.expand(false)),
        agg_icmp_fields = join(NetworkAggField::Icmp.expand(false)),
        agg_icmp6_fields = join(NetworkAggField::Icmp6.expand(false)),
        default_fields = join(DEFAULT_NETWORK_FIELDS.to_owned()),
    )
});

/// Represents the tcp and udp sub-models of the network model.
#[derive(
    Clone,
    Debug,
    PartialEq,
    below_derive::EnumFromStr,
    below_derive::EnumToString
)]
pub enum TransportAggField {
    Tcp,
    Udp,
    Udp6,
}

impl AggField<NetworkModelFieldId> for TransportAggField {
    fn expand(&self, _detail: bool) -> Vec<NetworkModelFieldId> {
        use model::NetworkModelFieldId as FieldId;
        match self {
            Self::Tcp => model::TcpModelFieldId::unit_variant_iter()
                .map(FieldId::Tcp)
                .collect(),
            Self::Udp => model::UdpModelFieldId::unit_variant_iter()
                .map(FieldId::Udp)
                .collect(),
            Self::Udp6 => model::Udp6ModelFieldId::unit_variant_iter()
                .map(FieldId::Udp6)
                .collect(),
        }
    }
}

pub type TransportOptionField = DumpOptionField<NetworkModelFieldId, TransportAggField>;

pub static DEFAULT_TRANSPORT_FIELDS: &[TransportOptionField] = &[
    DumpOptionField::Unit(DumpField::Common(CommonField::Datetime)),
    DumpOptionField::Agg(TransportAggField::Tcp),
    DumpOptionField::Agg(TransportAggField::Udp),
    DumpOptionField::Agg(TransportAggField::Udp6),
    DumpOptionField::Unit(DumpField::Common(CommonField::Timestamp)),
];

const TRANSPORT_ABOUT: &str = "Dump the transport layer stats including tcp and udp";

/// Generated about message for Transport dump so supported fields are up-to-date.
static TRANSPORT_LONG_ABOUT: Lazy<String> = Lazy::new(|| {
    format!(
        r#"{about}

********************** Available fields **********************

{common_fields}, and expanded fields below.

********************** Aggregated fields **********************

* tcp: includes [{agg_tcp_fields}].

* udp: includes [{agg_udp_fields}].

* udp6: includes [{agg_udp6_fields}].

* --detail: no effect.

* --default: includes [{default_fields}].

* --everything: includes everything (equivalent to --default --detail).

********************** Example Commands **********************

Example:

$ below dump transport -b "08:30:00" -e "08:30:30" -f tcp udp -O json

"#,
        about = TRANSPORT_ABOUT,
        common_fields = join(CommonField::unit_variant_iter()),
        agg_tcp_fields = join(TransportAggField::Tcp.expand(false)),
        agg_udp_fields = join(TransportAggField::Udp.expand(false)),
        agg_udp6_fields = join(TransportAggField::Udp6.expand(false)),
        default_fields = join(DEFAULT_TRANSPORT_FIELDS.to_owned()),
    )
});

make_option! (OutputFormat {
    "raw": Raw,
    "csv": Csv,
    "json": Json,
    "kv": KeyVal,
});

#[derive(Debug, StructOpt, Default, Clone)]
pub struct GeneralOpt {
    /// Show all top layer fields. If --default is specified, it overrides any specified fields via --fields.
    #[structopt(long)]
    pub default: bool,
    /// Show all fields. If --everything is specified, --fields and --default are overridden.
    #[structopt(long)]
    pub everything: bool,
    /// Show more infomation other than default.
    #[structopt(short, long)]
    pub detail: bool,
    /// Begin time, same format as replay
    #[structopt(long, short)]
    pub begin: String,
    /// End time, same format as replay
    #[structopt(long, short)]
    pub end: Option<String>,
    /// Take a regex and apply to --select selected field. See command level doc for example.
    #[structopt(long, short = "F")]
    pub filter: Option<Regex>,
    /// Sort (lower to higher) by --select selected field. See command level doc for example.
    #[structopt(long)]
    pub sort: bool,
    /// Sort (higher to lower) by --select selected field. See command level doc for example.
    #[structopt(long)]
    pub rsort: bool,
    // display top N field. See command level doc for example.
    #[structopt(long, default_value = "0")]
    pub top: u32,
    /// Repeat title, for each N line, it will render a line of title. Only for raw output format.
    #[structopt(long = "repeat-title")]
    pub repeat_title: Option<usize>,
    /// Output format. Choose from raw, csv, kv, json. Default to raw
    #[structopt(long, short = "O")]
    pub output_format: Option<OutputFormat>,
    /// Output destination, default to stdout.
    #[structopt(long, short)]
    pub output: Option<String>,
    /// Disable title in raw or csv format output
    #[structopt(long)]
    pub disable_title: bool,
    /// Days adjuster, same as -r option in replay.
    #[structopt(short = "r")]
    pub yesterdays: Option<String>,
    /// Line break symbol between samples
    #[structopt(long)]
    pub br: Option<String>,
    /// Dump raw data without units or conversion
    #[structopt(long)]
    pub raw: bool,
}

#[derive(Debug, StructOpt, Clone)]
pub enum DumpCommand {
    #[structopt(about = SYSTEM_ABOUT, long_about = SYSTEM_LONG_ABOUT.as_str())]
    System {
        /// Select which fields to display and in what order.
        #[structopt(short, long)]
        fields: Option<Vec<SystemOptionField>>,
        #[structopt(flatten)]
        opts: GeneralOpt,
        /// Saved pattern in the dumprc file under [system] section.
        #[structopt(long, short, conflicts_with("fields"))]
        pattern: Option<String>,
    },
    #[structopt(about = DISK_ABOUT, long_about = DISK_LONG_ABOUT.as_str())]
    Disk {
        /// Select which fields to display and in what order.
        #[structopt(short, long)]
        fields: Option<Vec<DiskOptionField>>,
        #[structopt(flatten)]
        opts: GeneralOpt,
        /// Select field for operation, use with --sort, --rsort, --filter, --top
        #[structopt(long, short)]
        select: Option<SingleDiskModelFieldId>,
        /// Saved pattern in the dumprc file under [disk] section.
        #[structopt(long, short, conflicts_with("fields"))]
        pattern: Option<String>,
    },
    #[structopt(about = PROCESS_ABOUT, long_about = PROCESS_LONG_ABOUT.as_str())]
    Process {
        /// Select which fields to display and in what order.
        #[structopt(short, long)]
        fields: Option<Vec<ProcessOptionField>>,
        #[structopt(flatten)]
        opts: GeneralOpt,
        /// Select field for operation, use with --sort, --rsort, --filter, --top
        #[structopt(long, short)]
        select: Option<SingleProcessModelFieldId>,
        /// Saved pattern in the dumprc file under [process] section.
        #[structopt(long, short, conflicts_with("fields"))]
        pattern: Option<String>,
    },
    #[structopt(about = CGROUP_ABOUT, long_about = CGROUP_LONG_ABOUT.as_str())]
    Cgroup {
        /// Select which fields to display and in what order.
        #[structopt(short, long)]
        fields: Option<Vec<CgroupOptionField>>,
        #[structopt(flatten)]
        opts: GeneralOpt,
        /// Select field for operation, use with --sort, --rsort, --filter, --top
        #[structopt(long, short)]
        select: Option<SingleCgroupModelFieldId>,
        /// Saved pattern in the dumprc file under [cgroup] section.
        #[structopt(long, short, conflicts_with("fields"))]
        pattern: Option<String>,
    },
    #[structopt(about = IFACE_ABOUT, long_about = IFACE_LONG_ABOUT.as_str())]
    Iface {
        /// Select which fields to display and in what order.
        #[structopt(short, long)]
        fields: Option<Vec<IfaceOptionField>>,
        #[structopt(flatten)]
        opts: GeneralOpt,
        /// Select field for operation, use with --filter
        #[structopt(long, short)]
        select: Option<SingleNetModelFieldId>,
        /// Saved pattern in the dumprc file under [iface] section.
        #[structopt(long, short, conflicts_with("fields"))]
        pattern: Option<String>,
    },
    #[structopt(about = NETWORK_ABOUT, long_about = NETWORK_LONG_ABOUT.as_str())]
    Network {
        /// Select which fields to display and in what order.
        #[structopt(short, long)]
        fields: Option<Vec<NetworkOptionField>>,
        #[structopt(flatten)]
        opts: GeneralOpt,
        /// Saved pattern in the dumprc file under [network] section.
        #[structopt(long, short, conflicts_with("fields"))]
        pattern: Option<String>,
    },
    #[structopt(about = TRANSPORT_ABOUT, long_about = TRANSPORT_LONG_ABOUT.as_str())]
    Transport {
        /// Select which fields to display and in what order.
        #[structopt(short, long)]
        fields: Option<Vec<TransportOptionField>>,
        #[structopt(flatten)]
        opts: GeneralOpt,
        /// Saved pattern in the dumprc file under [transport] section.
        #[structopt(long, short, conflicts_with("fields"))]
        pattern: Option<String>,
    },
}
