//! This module contains traits and structs related to beliefs.

use std::collections::HashMap;

use simple_error::{bail, SimpleResult};

use crate::behaviour::Behaviour;

/// This is a [`Belief`].
pub struct Belief {
    /// The name of the [`Belief`].
    pub name: String,
    /// The name of the [`Belief`].
    pub uuid: uuid::Uuid,
    /// The perception that a [`Behaviour`] is driven by the [`Belief`].
    /// The [`Behaviour`] is identified by it's [`uuid::Uuid`].
    perception: HashMap<uuid::Uuid, f64>,
}

impl Belief {
    /// Create a new [`Belief`].
    /// It takes just a `name` as an argument.
    /// The `uuid` is randomly generated.
    ///
    /// ```rust
    /// use belief_spread::belief::Belief;
    /// let x = Belief::new("name".to_string());
    /// ```
    pub fn new(name: String) -> Self {
        Self::new_with_uuid(name, uuid::Uuid::new_v4())
    }

    /// Create a new [`Belief`].
    /// It takes just a `name` and `uuid` as an argument.
    ///
    /// ```rust
    /// use belief_spread::belief::Belief;
    /// let x = Belief::new_with_uuid(
    ///     "name".to_string(),
    ///     uuid::Uuid::new_v4()
    /// );
    /// ```
    pub fn new_with_uuid(name: String, uuid: uuid::Uuid) -> Self {
        Self {
            name,
            uuid,
            perception: HashMap::new(),
        }
    }

    /// Gets the perception that the [`Behaviour`] is driven by this [`Belief`].
    /// Returns [`None`] if no perception is defined.
    ///
    /// ```rust
    /// use belief_spread::{belief::Belief, behaviour::Behaviour};
    /// let behaviour = Behaviour::default();
    /// let belief = Belief::new("b1".to_string());
    /// assert_eq!(belief.get_perception(&behaviour), None);
    /// ```
    pub fn get_perception(&self, behaviour: &Behaviour) -> Option<f64> {
        self.perception.get(&behaviour.uuid).copied()
    }

    /// Sets the perception that the [`Behaviour`] is driven by this [`Belief`].
    /// Sets this to `new_perception`. If this value is [`None`], the perception
    /// is removed if it exists. Returns [`Err(simple_error::SimpleError)`] if
    /// the new perception is not in the range [-1, +1].
    ///
    /// ```rust
    /// use belief_spread::{belief::Belief, behaviour::Behaviour};
    /// let behaviour = Behaviour::default();
    /// let mut belief = Belief::new("b1".to_string());
    /// assert_eq!(belief.get_perception(&behaviour), None);
    /// assert_eq!(belief.set_perception(&behaviour, Some(0.1)), Ok(()));
    /// assert_eq!(belief.get_perception(&behaviour), Some(0.1));
    /// assert_eq!(belief.set_perception(&behaviour, Some(2.0)),
    ///     Err(simple_error::simple_error!("new_perception is greater than 1")));
    /// assert_eq!(belief.get_perception(&behaviour), Some(0.1));
    /// assert_eq!(belief.set_perception(&behaviour, Some(-2.0)),
    ///     Err(simple_error::simple_error!("new_perception is less than -1")));
    /// assert_eq!(belief.set_perception(&behaviour, None), Ok(()));
    /// assert_eq!(belief.get_perception(&behaviour), None);
    /// ```
    pub fn set_perception(
        &mut self,
        behaviour: &Behaviour,
        new_perception: Option<f64>,
    ) -> SimpleResult<()> {
        match new_perception {
            None => {
                self.perception.remove(&behaviour.uuid);
                Ok(())
            }
            Some(x) if x > 1.0 => bail!("new_perception is greater than 1"),
            Some(x) if x < -1.0 => bail!("new_perception is less than -1"),
            Some(x) => {
                self.perception.insert(behaviour.uuid, x);
                Ok(())
            }
        }
    }
}

#[cfg(test)]
mod tests {
    use super::Belief;
    use crate::behaviour::Behaviour;
    use simple_error::simple_error;

    #[test]
    fn new_assigns_name() {
        let x = Belief::new("name".to_string());
        assert_eq!(x.name, "name");
    }

    #[test]
    fn new_with_uuid_assigns_uuid() {
        let u = uuid::Uuid::new_v4();
        let x = Belief::new_with_uuid("x".to_string(), u);
        assert_eq!(x.uuid, u);
        assert_eq!(x.name, "x");
    }
    #[test]
    fn new_assigns_random_uuid() {
        let x1 = Belief::new("x1".to_string());
        let x2 = Belief::new("x2".to_string());
        assert_eq!(x1.name, "x1");
        assert_eq!(x2.name, "x2");
        assert_ne!(x1.uuid, x2.uuid);
    }

    #[test]
    fn get_perception_when_not_exists_returns_none() {
        let behaviour = Behaviour::default();
        let belief = Belief::new("name".to_string());
        assert_eq!(belief.get_perception(&behaviour), None);
    }

    #[test]
    fn set_perception_when_valid() {
        let behaviour = Behaviour::default();

        let mut belief = Belief::new("name".to_string());
        assert_eq!(belief.get_perception(&behaviour), None);
        assert_eq!(belief.set_perception(&behaviour, Some(0.5)), Ok(()));
        assert_eq!(belief.get_perception(&behaviour), Some(0.5));
    }

    #[test]
    fn set_perception_when_valid_delete() {
        let behaviour = Behaviour::default();
        let mut belief = Belief::new("name".to_string());
        assert_eq!(belief.get_perception(&behaviour), None);
        assert_eq!(belief.set_perception(&behaviour, Some(0.5)), Ok(()));
        assert_eq!(belief.get_perception(&behaviour), Some(0.5));
        assert_eq!(belief.set_perception(&behaviour, None), Ok(()));
        assert_eq!(belief.get_perception(&behaviour), None);
    }

    #[test]
    fn set_perception_when_too_high() {
        let behaviour = Behaviour::default();
        let mut belief = Belief::new("name".to_string());
        assert_eq!(belief.get_perception(&behaviour), None);
        assert_eq!(
            belief.set_perception(&behaviour, Some(2.0)),
            Err(simple_error!("new_perception is greater than 1"))
        );
    }

    #[test]
    fn set_perception_when_too_low() {
        let behaviour = Behaviour::default();
        let mut belief = Belief::new("name".to_string());
        assert_eq!(belief.get_perception(&behaviour), None);
        assert_eq!(
            belief.set_perception(&behaviour, Some(-2.0)),
            Err(simple_error!("new_perception is less than -1"))
        );
    }
}
