//! This module contains traits and structs related to beliefs.

use std::collections::HashMap;

use mockall::*;
use simple_error::{bail, SimpleResult};

use crate::{behaviour::Behaviour, named::Named, uuidd::UUIDd};

/// A [`Belief`] is [`Named`] and [`UUIDd`]
#[automock]
pub trait Belief {
    /// Gets the amount that an agent performing [`Behaviour`] B can be
    /// perceived to be believing the belief [`self`].
    ///
    /// This value should be between -1 and +1.
    ///
    /// ```rust
    /// use mockall::predicate::*;
    /// use mockall::*;
    ///
    /// use belief_spread::{behaviour::Behaviour, belief::{Belief, MockBelief}, uuidd::UUIDd};
    ///
    /// mock! {
    ///     B {}
    ///     impl Behaviour for B {}
    ///     impl UUIDd for B {
    ///         fn get_uuid(&self) -> &uuid::Uuid;
    ///         fn set_uuid(&mut self, new_uuid: uuid::Uuid);
    ///     }
    /// }
    ///
    /// let mut mock_belief = MockBelief::new();
    /// let mock_behaviour = MockB::new();
    /// mock_belief.expect_get_perception::<MockB>()
    ///     .returning(|_| Some(1.0));
    ///
    /// assert_eq!(Some(1.0), mock_belief.get_perception(&mock_behaviour));
    /// ```
    fn get_perception<B: 'static + Behaviour + UUIDd>(&self, behaviour: &B) -> Option<f64>;
    /// Sets the amount that an agent performing [`Behaviour`] B can be
    /// perceived to be believing the belief [`self`] to `new_perception`.
    /// Perception must be between -1 and +1. If new_perception is [`None`],
    /// then the perception is deleted.
    ///
    /// ```rust
    /// use mockall::predicate::*;
    /// use mockall::*;
    ///
    /// use belief_spread::{behaviour::Behaviour, belief::{Belief, MockBelief}, uuidd::UUIDd};
    ///
    /// mock! {
    ///     B {}
    ///     impl Behaviour for B {}
    ///     impl UUIDd for B {
    ///         fn get_uuid(&self) -> &uuid::Uuid;
    ///         fn set_uuid(&mut self, new_uuid: uuid::Uuid);
    ///     }
    /// }
    ///
    /// let mut mock_belief = MockBelief::new();
    /// let mock_behaviour = MockB::new();
    /// mock_belief.expect_set_perception::<MockB>()
    ///     .returning(|_, _| Ok(()));
    /// mock_belief.set_perception(&mock_behaviour, Some(0.0));
    /// ```
    fn set_perception<B: 'static + Behaviour + UUIDd>(
        &mut self,
        behaviour: &B,
        new_perception: Option<f64>,
    ) -> SimpleResult<()>;
}

/// This is an implementation of [`Belief`].
pub struct BasicBelief {
    /// The name of the [`BasicBelief`].
    name: String,
    /// The name of the [`BasicBelief`].
    uuid: uuid::Uuid,
    /// The perception that a [`Behaviour`] is driven by the [`Belief`].
    /// The [`Behaviour`] is identified by it's [`uuid::Uuid`].
    perception: HashMap<uuid::Uuid, f64>,
}

impl BasicBelief {
    /// Create a new [`BasicBelief`].
    ///
    /// This is an implementation of [`Belief`].
    /// It takes just a `name` as an argument.
    /// The `uuid` is randomly generated.
    ///
    /// ```rust
    /// use belief_spread::belief::BasicBelief;
    /// let x = BasicBelief::new("name".to_string());
    /// ```
    pub fn new(name: String) -> Self {
        Self::new_with_uuid(name, uuid::Uuid::new_v4())
    }

    /// Create a new [`BasicBelief`].
    ///
    /// This is an implementation of [`Belief`].
    /// It takes just a `name` and `uuid` as an argument.
    ///
    /// ```rust
    /// use belief_spread::belief::BasicBelief;
    /// let x = BasicBelief::new_with_uuid(
    ///     "name".to_string(),
    ///     uuid::Uuid::new_v4()
    /// );
    /// ```
    pub fn new_with_uuid(name: String, uuid: uuid::Uuid) -> Self {
        Self {
            name,
            uuid,
            perception: HashMap::new(),
        }
    }
}

impl Belief for BasicBelief {
    fn get_perception<B: 'static + Behaviour + UUIDd>(&self, behaviour: &B) -> Option<f64> {
        self.perception.get(behaviour.get_uuid()).copied()
    }

    fn set_perception<B: 'static + Behaviour + UUIDd>(
        &mut self,
        behaviour: &B,
        new_perception: Option<f64>,
    ) -> SimpleResult<()> {
        match new_perception {
            None => {
                self.perception.remove(behaviour.get_uuid());
                Ok(())
            }
            Some(x) if x > 1.0 => bail!("new_perception is greater than 1"),
            Some(x) if x < -1.0 => bail!("new_perception is less than -1"),
            Some(x) => {
                self.perception.insert(*behaviour.get_uuid(), x);
                Ok(())
            }
        }
    }
}

impl Named for BasicBelief {
    fn get_name(&self) -> &str {
        self.name.as_str()
    }

    fn set_name(&mut self, name: String) {
        self.name = name;
    }
}

impl UUIDd for BasicBelief {
    fn get_uuid(&self) -> &uuid::Uuid {
        &self.uuid
    }

    fn set_uuid(&mut self, new_uuid: uuid::Uuid) {
        self.uuid = new_uuid;
    }
}

#[cfg(test)]
mod tests {
    use mockall::mock;

    use super::BasicBelief;
    use crate::{behaviour::Behaviour, belief::Belief, named::Named, uuidd::UUIDd};
    use simple_error::simple_error;

    #[test]
    fn new_assigns_name_then_get_name() {
        let x = BasicBelief::new("name".to_string());
        assert_eq!(x.get_name(), "name");
    }

    #[test]
    fn new_assigns_name_then_set_name_then_get_name() {
        let mut x = BasicBelief::new("name".to_string());
        assert_eq!(x.get_name(), "name");
        x.set_name("name2".to_string());
        assert_eq!(x.get_name(), "name2");
    }

    #[test]
    fn new_with_uuid_then_get_uuid() {
        let u = uuid::Uuid::new_v4();
        let x = BasicBelief::new_with_uuid("x".to_string(), u);
        assert_eq!(x.get_uuid(), &u);
        assert_eq!(x.get_name(), "x");
    }

    #[test]
    fn new_with_uuid_then_set_uuid_then_get_uuid() {
        let u1 = uuid::Uuid::new_v4();
        let mut x = BasicBelief::new_with_uuid("x".to_string(), u1);
        assert_eq!(x.get_uuid(), &u1);
        assert_eq!(x.get_name(), "x");
        let u2 = uuid::Uuid::new_v4();
        x.set_uuid(u2);
        assert_eq!(x.get_uuid(), &u2);
        assert_eq!(x.get_name(), "x");
    }

    #[test]
    fn new_assigns_random_uuid() {
        let x1 = BasicBelief::new("x1".to_string());
        let x2 = BasicBelief::new("x2".to_string());
        assert_eq!(x1.get_name(), "x1");
        assert_eq!(x2.get_name(), "x2");
        assert_ne!(x1.get_uuid(), x2.get_uuid());
    }

    #[test]
    fn get_perception_when_not_exists_returns_none() {
        mock! {
            B {}
            impl Behaviour for B {}
            impl UUIDd for B {
                fn get_uuid(&self) -> &uuid::Uuid;
                fn set_uuid(&mut self, new_uuid: uuid::Uuid);
            }
        }

        let mut mock_behaviour = MockB::new();
        let uuid_v = uuid::Uuid::new_v4();
        mock_behaviour
            .expect_get_uuid()
            .times(1)
            .return_const(uuid_v);

        let belief = BasicBelief::new("name".to_string());
        assert_eq!(belief.get_perception(&mock_behaviour), None);
    }

    #[test]
    fn set_perception_when_valid() {
        mock! {
            B {}
            impl Behaviour for B {}
            impl UUIDd for B {
                fn get_uuid(&self) -> &uuid::Uuid;
                fn set_uuid(&mut self, new_uuid: uuid::Uuid);
            }
        }

        let mut mock_behaviour = MockB::new();
        let uuid_v = uuid::Uuid::new_v4();
        mock_behaviour
            .expect_get_uuid()
            .times(3)
            .return_const(uuid_v);

        let mut belief = BasicBelief::new("name".to_string());
        assert_eq!(belief.get_perception(&mock_behaviour), None);
        assert_eq!(belief.set_perception(&mock_behaviour, Some(0.5)), Ok(()));
        assert_eq!(belief.get_perception(&mock_behaviour), Some(0.5));
    }

    #[test]
    fn set_perception_when_valid_delete() {
        mock! {
            B {}
            impl Behaviour for B {}
            impl UUIDd for B {
                fn get_uuid(&self) -> &uuid::Uuid;
                fn set_uuid(&mut self, new_uuid: uuid::Uuid);
            }
        }

        let mut mock_behaviour = MockB::new();
        let uuid_v = uuid::Uuid::new_v4();
        mock_behaviour
            .expect_get_uuid()
            .times(5)
            .return_const(uuid_v);

        let mut belief = BasicBelief::new("name".to_string());
        assert_eq!(belief.get_perception(&mock_behaviour), None);
        assert_eq!(belief.set_perception(&mock_behaviour, Some(0.5)), Ok(()));
        assert_eq!(belief.get_perception(&mock_behaviour), Some(0.5));
        assert_eq!(belief.set_perception(&mock_behaviour, None), Ok(()));
        assert_eq!(belief.get_perception(&mock_behaviour), None);
    }

    #[test]
    fn set_perception_when_too_high() {
        mock! {
            B {}
            impl Behaviour for B {}
            impl UUIDd for B {
                fn get_uuid(&self) -> &uuid::Uuid;
                fn set_uuid(&mut self, new_uuid: uuid::Uuid);
            }
        }

        let mut mock_behaviour = MockB::new();
        let uuid_v = uuid::Uuid::new_v4();
        mock_behaviour
            .expect_get_uuid()
            .times(1)
            .return_const(uuid_v);

        let mut belief = BasicBelief::new("name".to_string());
        assert_eq!(belief.get_perception(&mock_behaviour), None);
        assert_eq!(
            belief.set_perception(&mock_behaviour, Some(2.0)),
            Err(simple_error!("new_perception is greater than 1"))
        );
    }

    #[test]
    fn set_perception_when_too_low() {
        mock! {
            B {}
            impl Behaviour for B {}
            impl UUIDd for B {
                fn get_uuid(&self) -> &uuid::Uuid;
                fn set_uuid(&mut self, new_uuid: uuid::Uuid);
            }
        }

        let mut mock_behaviour = MockB::new();
        let uuid_v = uuid::Uuid::new_v4();
        mock_behaviour
            .expect_get_uuid()
            .times(1)
            .return_const(uuid_v);

        let mut belief = BasicBelief::new("name".to_string());
        assert_eq!(belief.get_perception(&mock_behaviour), None);
        assert_eq!(
            belief.set_perception(&mock_behaviour, Some(-2.0)),
            Err(simple_error!("new_perception is less than -1"))
        );
    }
}
