#![allow(clippy::unreadable_literal)]
use beatrice::internal::{DateTime, FormatTime};
use std::time::{Duration, SystemTime};

const MIN: u64 = 60;
const HOUR: u64 = 60 * MIN;
const DAY: u64 = 24 * HOUR;

#[test]
fn date_time_new() {
    for (expected, epoch_seconds) in [
        ((1970, 1, 1, 0, 0, 0), 0),
        ((1970, 1, 1, 0, 0, 1), 1),
        ((1970, 1, 1, 0, 0, 59), 59),
        ((1970, 1, 1, 0, 1, 0), 60),
        ((1970, 1, 1, 0, 1, 1), 61),
        ((1970, 1, 1, 0, 59, 0), 59 * 60),
        ((1970, 1, 1, 1, 0, 0), 60 * 60),
        ((1970, 1, 1, 23, 0, 0), 23 * 60 * 60),
        ((1970, 1, 1, 23, 59, 59), 86400 - 1),
        ((1970, 1, 2, 0, 0, 0), 86400),
        ((1970, 1, 2, 0, 0, 1), 86400 + 1),
        ((1970, 1, 31, 23, 59, 59), 31 * 86400 - 1),
        ((1970, 2, 1, 0, 0, 0), 31 * 86400),
        ((1970, 3, 1, 0, 0, 0), 59 * 86400),
        ((1970, 12, 31, 23, 59, 59), 31535999),
        ((1971, 1, 1, 0, 0, 0), 31536000),
        ((1972, 6, 30, 23, 59, 59), 78796799),
        ((1972, 7, 1, 0, 0, 0), 78796800),
        ((2022, 3, 30, 7, 29, 33), 1648625373),
        ((2022, 3, 30, 7, 29, 33), 1648625373),
        ((2100, 2, 28, 23, 59, 59), 4107542399),
        ((2100, 3, 1, 0, 0, 0), 4107542400),
    ] {
        let dt = DateTime::new(epoch_seconds);
        assert_eq!(
            expected,
            (dt.year, dt.month, dt.day, dt.hour, dt.min, dt.sec)
        );
    }
}

#[test]
fn date_time_add() {
    for (initial, seconds_to_add, expected) in [
        ((1970, 1, 1, 0, 0, 0), 0, (1970, 1, 1, 0, 0, 0)),
        ((1970, 1, 1, 0, 0, 0), 1, (1970, 1, 1, 0, 0, 1)),
        ((2004, 2, 28, 23, 59, 59), 1, (2004, 2, 29, 0, 0, 0)),
        ((2100, 2, 28, 23, 59, 59), 1, (2100, 3, 1, 0, 0, 0)),
        ((2000, 2, 28, 23, 59, 59), 1, (2000, 2, 29, 0, 0, 0)),
        ((2004, 2, 28, 0, 0, 0), 365 * DAY, (2005, 2, 27, 0, 0, 0)),
        ((2100, 2, 28, 0, 0, 0), 365 * DAY, (2101, 2, 28, 0, 0, 0)),
        ((2000, 2, 28, 0, 0, 0), 365 * DAY, (2001, 2, 27, 0, 0, 0)),
        ((2004, 2, 28, 0, 0, 0), 366 * DAY, (2005, 2, 28, 0, 0, 0)),
        ((2100, 2, 28, 0, 0, 0), 366 * DAY, (2101, 3, 1, 0, 0, 0)),
        ((2000, 2, 28, 0, 0, 0), 366 * DAY, (2001, 2, 28, 0, 0, 0)),
        ((2004, 2, 28, 0, 0, 0), 367 * DAY, (2005, 3, 1, 0, 0, 0)),
        ((2100, 2, 28, 0, 0, 0), 367 * DAY, (2101, 3, 2, 0, 0, 0)),
        ((2000, 2, 28, 0, 0, 0), 367 * DAY, (2001, 3, 1, 0, 0, 0)),
        ((2004, 2, 28, 0, 0, 0), 1462 * DAY, (2008, 2, 29, 0, 0, 0)),
        ((2100, 2, 28, 0, 0, 0), 1462 * DAY, (2104, 3, 1, 0, 0, 0)),
        ((2000, 2, 28, 0, 0, 0), 1462 * DAY, (2004, 2, 29, 0, 0, 0)),
    ] {
        let dt = DateTime {
            year: initial.0,
            month: initial.1,
            day: initial.2,
            hour: initial.3,
            min: initial.4,
            sec: initial.5,
        } + Duration::from_secs(seconds_to_add);
        assert_eq!(
            expected,
            (dt.year, dt.month, dt.day, dt.hour, dt.min, dt.sec),
            "{:?} + {}",
            initial,
            seconds_to_add,
        );
    }
}

#[test]
fn test_iso8601_utc() {
    for (expected, epoch_seconds) in [
        ("1970-01-01T00:00:00Z", 0),
        ("2022-03-30T07:29:33Z", 1648625373),
        ("2100-02-28T23:59:59Z", 4107542399),
    ] {
        assert_eq!(
            expected,
            (SystemTime::UNIX_EPOCH + Duration::from_secs(epoch_seconds)).iso8601_utc()
        );
    }
}
