use std::ops::{Deref, DerefMut};

use chrono::{DateTime, Datelike, Local};
use serde::{de::Visitor, Deserialize, Serialize};

#[derive(Debug, Clone)]
pub struct LocalTime(pub DateTime<Local>);

impl LocalTime {
  pub fn to_date_string(&self) -> String {
    format!("{:02}-{:02}-{:04}", self.month(), self.day(), self.year())
  }
}

impl Deref for LocalTime {
  type Target = DateTime<Local>;

  fn deref(&self) -> &Self::Target {
    &self.0
  }
}

impl DerefMut for LocalTime {
  fn deref_mut(&mut self) -> &mut Self::Target {
    &mut self.0
  }
}

impl Serialize for LocalTime {
  fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
  where
    S: serde::Serializer,
  {
    serializer.serialize_str(&self.to_rfc3339())
  }
}

struct LocalTimeVisitor;

impl<'de> Visitor<'de> for LocalTimeVisitor {
  type Value = LocalTime;

  fn expecting(&self, formatter: &mut std::fmt::Formatter) -> std::fmt::Result {
    write!(formatter, "a string")
  }

  fn visit_str<E>(self, v: &str) -> Result<Self::Value, E>
  where
    E: serde::de::Error,
  {
    let parsed_time = match DateTime::parse_from_rfc3339(v) {
      Ok(v) => v,
      Err(err) => {
        return Err(E::custom(format!("failed to parse date. Err: {}", err)))
      }
    };
    Ok(LocalTime(DateTime::<Local>::from(parsed_time)))
  }
}

impl<'de> Deserialize<'de> for LocalTime {
  fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
  where
    D: serde::Deserializer<'de>,
  {
    deserializer.deserialize_str(LocalTimeVisitor)
  }
}
