use std::collections::HashMap;

/// ### Command Arg Parser
///
///	use [`parse_args`] to parse arg
///
/// example:
/// ```no_run
/// let parsed_args = parse_args(env::args());
/// ```
///
/// Output:
/// ParsedArgs {
///     options: {
///         "-c": Some(
///             "true",
///         ),
///         "-none": None,
///     },
///     additional_options: [
///         "--lol",
///     ],
///     normal_args: [
///         "todo",
///         "add",
///     ],
/// }
#[derive(Debug)]
pub struct ParsedArgs {
    pub options: HashMap<String, Option<String>>,
    pub additional_options: Vec<String>,
    pub normal_args: Vec<String>,
}

impl ParsedArgs {
    pub fn do_with_option(&self, option: &str, func: impl FnOnce(&String)) {
        if let Some(value) = self.options.get(option) {
            if let Some(value) = value {
                func(value);
            }
        }
    }
}

pub fn parse_args(mut args: Vec<String>) -> ParsedArgs {
    let mut parse_args = ParsedArgs {
        options: HashMap::new(),
        additional_options: Vec::new(),
        normal_args: Vec::new(),
    };

    while args.len() > 0 {
        if args[0].starts_with("--") {
            parse_args.additional_options.push(args.remove(0));
        } else if args[0].starts_with('-') && args.len() > 1 {
            parse_args
                .options
                .insert(args.remove(0), Some(args.remove(0)));
        } else if args[0].starts_with('-') && args.len() == 1 {
            parse_args.options.insert(args.remove(0), None);
        } else {
            parse_args.normal_args.push(args.remove(0));
        }
    }

    parse_args
}
