/*
 * Copyright (C) 2021 taylor.fish <contact@taylor.fish>
 *
 * This file is part of Base116.
 *
 * Base116 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Base116 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Base116. If not, see <https://www.gnu.org/licenses/>.
 */

use core::convert::TryFrom;
use core::ops::Range;

pub fn map<R>(mut orig: u16, ranges: R) -> Option<u16>
where
    R: IntoIterator<Item = Range<u16>>,
{
    for range in ranges {
        orig = match orig.checked_sub(range.end - range.start) {
            Some(n) => n,
            None => return Some(range.start + orig),
        }
    }
    None
}

pub fn unmap<R>(mapped: u16, ranges: R) -> Option<u16>
where
    R: IntoIterator<Item = Range<u16>>,
{
    let mut offset = 0;
    for range in ranges {
        if mapped < range.end {
            return Some(mapped.checked_sub(range.start)? + offset);
        }
        offset += range.end - range.start;
    }
    None
}

pub fn map_char<R>(orig: u16, ranges: R) -> Option<char>
where
    R: IntoIterator<Item = Range<u16>>,
{
    map(orig, ranges).map(u32::from).map(|n| {
        if let Some(c) = char::from_u32(n) {
            c
        } else {
            panic!("not a valid char: {} (check `ranges`)", n);
        }
    })
}

pub fn unmap_char<R>(mapped: char, ranges: R) -> Option<u16>
where
    R: IntoIterator<Item = Range<u16>>,
{
    unmap(u16::try_from(u32::from(mapped)).ok()?, ranges)
}

pub const RANGES1: [Range<u16>; 2] = [0x0009..0x000b, 0x0020..0x007f];

pub const RANGES2: [Range<u16>; 23] = [
    0x00a1..0x00ad,
    0x00ae..0x01f1,
    0x01f3..0x02c8,
    0x02c9..0x034f,
    0x0350..0x0378,
    0x037a..0x0380,
    0x0384..0x038b,
    0x038c..0x038d,
    0x038e..0x03a2,
    0x03a3..0x0530,
    0x0531..0x0557,
    0x0559..0x058b,
    0x058d..0x0590,
    0x0591..0x05c3,
    0x05c4..0x05c8,
    0x05d0..0x05eb,
    0x05ef..0x05f5,
    0x0606..0x061c,
    0x061d..0x06dd,
    0x06de..0x070e,
    0x0710..0x074b,
    0x074d..0x07b2,
    0x07c0..0x07c2,
];

pub const RANGES3: [Range<u16>; 358] = [
    0x0800..0x0816,
    0x081a..0x081b,
    0x0824..0x0825,
    0x0828..0x0829,
    0x0830..0x083f,
    0x0840..0x0859,
    0x085e..0x085f,
    0x0860..0x086b,
    0x0870..0x088f,
    0x08a0..0x08ca,
    0x0904..0x093a,
    0x093d..0x093e,
    0x0950..0x0951,
    0x0958..0x0962,
    0x0964..0x0981,
    0x0985..0x098d,
    0x098f..0x0991,
    0x0993..0x09a9,
    0x09aa..0x09b1,
    0x09b2..0x09b3,
    0x09b6..0x09ba,
    0x09bd..0x09be,
    0x09ce..0x09cf,
    0x09dc..0x09de,
    0x09df..0x09e2,
    0x09e6..0x09fe,
    0x0a05..0x0a0b,
    0x0a0f..0x0a11,
    0x0a13..0x0a29,
    0x0a2a..0x0a31,
    0x0a32..0x0a34,
    0x0a35..0x0a37,
    0x0a38..0x0a3a,
    0x0a59..0x0a5d,
    0x0a5e..0x0a5f,
    0x0a66..0x0a70,
    0x0a72..0x0a75,
    0x0a76..0x0a77,
    0x0a85..0x0a8e,
    0x0a8f..0x0a92,
    0x0a93..0x0aa9,
    0x0aaa..0x0ab1,
    0x0ab2..0x0ab4,
    0x0ab5..0x0aba,
    0x0abd..0x0abe,
    0x0ad0..0x0ad1,
    0x0ae0..0x0ae2,
    0x0ae6..0x0af2,
    0x0af9..0x0afa,
    0x0b05..0x0b0d,
    0x0b0f..0x0b11,
    0x0b13..0x0b29,
    0x0b2a..0x0b31,
    0x0b32..0x0b34,
    0x0b35..0x0b3a,
    0x0b3d..0x0b3e,
    0x0b5c..0x0b5e,
    0x0b5f..0x0b62,
    0x0b66..0x0b78,
    0x0b83..0x0b84,
    0x0b85..0x0b8b,
    0x0b8e..0x0b91,
    0x0b92..0x0b96,
    0x0b99..0x0b9b,
    0x0b9c..0x0b9d,
    0x0b9e..0x0ba0,
    0x0ba3..0x0ba5,
    0x0ba8..0x0bab,
    0x0bae..0x0bba,
    0x0bd0..0x0bd1,
    0x0be6..0x0bfb,
    0x0c05..0x0c0d,
    0x0c0e..0x0c11,
    0x0c12..0x0c29,
    0x0c2a..0x0c3a,
    0x0c3d..0x0c3e,
    0x0c58..0x0c5b,
    0x0c5d..0x0c5e,
    0x0c60..0x0c62,
    0x0c66..0x0c70,
    0x0c77..0x0c81,
    0x0c84..0x0c8d,
    0x0c8e..0x0c91,
    0x0c92..0x0ca9,
    0x0caa..0x0cb4,
    0x0cb5..0x0cba,
    0x0cbd..0x0cbe,
    0x0cdd..0x0cdf,
    0x0ce0..0x0ce2,
    0x0ce6..0x0cf0,
    0x0cf1..0x0cf3,
    0x0d04..0x0d0d,
    0x0d0e..0x0d11,
    0x0d12..0x0d3b,
    0x0d3d..0x0d3e,
    0x0d4e..0x0d50,
    0x0d54..0x0d57,
    0x0d58..0x0d62,
    0x0d66..0x0d80,
    0x0d85..0x0d97,
    0x0d9a..0x0db2,
    0x0db3..0x0dbc,
    0x0dbd..0x0dbe,
    0x0dc0..0x0dc7,
    0x0de6..0x0df0,
    0x0df4..0x0df5,
    0x0e01..0x0e31,
    0x0e32..0x0e34,
    0x0e3f..0x0e47,
    0x0e4f..0x0e5c,
    0x0e81..0x0e83,
    0x0e84..0x0e85,
    0x0e86..0x0e8b,
    0x0e8c..0x0ea4,
    0x0ea5..0x0ea6,
    0x0ea7..0x0eb1,
    0x0eb2..0x0eb4,
    0x0ebd..0x0ebe,
    0x0ec0..0x0ec5,
    0x0ec6..0x0ec7,
    0x0ed0..0x0eda,
    0x0edc..0x0ee0,
    0x0f00..0x0f18,
    0x0f1a..0x0f35,
    0x0f36..0x0f37,
    0x0f38..0x0f39,
    0x0f3a..0x0f3e,
    0x0f40..0x0f48,
    0x0f49..0x0f6d,
    0x0f85..0x0f86,
    0x0f88..0x0f8d,
    0x0fbe..0x0fc6,
    0x0fc7..0x0fcd,
    0x0fce..0x0fdb,
    0x1000..0x102b,
    0x103f..0x1056,
    0x105a..0x105e,
    0x1061..0x1062,
    0x1065..0x1067,
    0x106e..0x1071,
    0x1075..0x1082,
    0x108e..0x108f,
    0x1090..0x109a,
    0x109e..0x10c6,
    0x10c7..0x10c8,
    0x10cd..0x10ce,
    0x10d0..0x115f,
    0x1161..0x1249,
    0x124a..0x124e,
    0x1250..0x1257,
    0x1258..0x1259,
    0x125a..0x125e,
    0x1260..0x1289,
    0x128a..0x128e,
    0x1290..0x12b1,
    0x12b2..0x12b6,
    0x12b8..0x12bf,
    0x12c0..0x12c1,
    0x12c2..0x12c6,
    0x12c8..0x12d7,
    0x12d8..0x1311,
    0x1312..0x1316,
    0x1318..0x135b,
    0x1360..0x137d,
    0x1380..0x139a,
    0x13a0..0x13f6,
    0x13f8..0x13fe,
    0x1400..0x1680,
    0x1681..0x169d,
    0x16a0..0x16f9,
    0x1700..0x1712,
    0x171f..0x1732,
    0x1735..0x1737,
    0x1740..0x1752,
    0x1760..0x176d,
    0x176e..0x1771,
    0x1780..0x17b4,
    0x17d4..0x17dd,
    0x17e0..0x17ea,
    0x17f0..0x17fa,
    0x1800..0x180b,
    0x1810..0x181a,
    0x1820..0x1879,
    0x1880..0x1885,
    0x1887..0x18a9,
    0x18aa..0x18ab,
    0x18b0..0x18f6,
    0x1900..0x191f,
    0x1940..0x1941,
    0x1944..0x196e,
    0x1970..0x1975,
    0x1980..0x19ac,
    0x19b0..0x19ca,
    0x19d0..0x19db,
    0x19de..0x1a17,
    0x1a1e..0x1a55,
    0x1a80..0x1a8a,
    0x1a90..0x1a9a,
    0x1aa0..0x1aae,
    0x1b05..0x1b34,
    0x1b45..0x1b4d,
    0x1b50..0x1b6b,
    0x1b74..0x1b7f,
    0x1b83..0x1ba1,
    0x1bae..0x1be6,
    0x1bfc..0x1c24,
    0x1c3b..0x1c4a,
    0x1c4d..0x1c89,
    0x1c90..0x1cbb,
    0x1cbd..0x1cc8,
    0x1cd3..0x1cd4,
    0x1ce9..0x1ced,
    0x1cee..0x1cf4,
    0x1cf5..0x1cf7,
    0x1cfa..0x1cfb,
    0x1d00..0x1dc0,
    0x1e00..0x1f16,
    0x1f18..0x1f1e,
    0x1f20..0x1f46,
    0x1f48..0x1f4e,
    0x1f50..0x1f58,
    0x1f59..0x1f5a,
    0x1f5b..0x1f5c,
    0x1f5d..0x1f5e,
    0x1f5f..0x1f7e,
    0x1f80..0x1fb5,
    0x1fb6..0x1fc5,
    0x1fc6..0x1fd4,
    0x1fd6..0x1fdc,
    0x1fdd..0x1ff0,
    0x1ff2..0x1ff5,
    0x1ff6..0x1fff,
    0x2010..0x2028,
    0x2030..0x205f,
    0x2070..0x2072,
    0x2074..0x208f,
    0x2090..0x209d,
    0x20a0..0x20c1,
    0x2100..0x218c,
    0x2190..0x2427,
    0x2440..0x244b,
    0x2460..0x2b74,
    0x2b76..0x2b96,
    0x2b97..0x2cef,
    0x2cf2..0x2cf4,
    0x2cf9..0x2d26,
    0x2d27..0x2d28,
    0x2d2d..0x2d2e,
    0x2d30..0x2d68,
    0x2d6f..0x2d71,
    0x2d80..0x2d97,
    0x2da0..0x2da7,
    0x2da8..0x2daf,
    0x2db0..0x2db7,
    0x2db8..0x2dbf,
    0x2dc0..0x2dc7,
    0x2dc8..0x2dcf,
    0x2dd0..0x2dd7,
    0x2dd8..0x2ddf,
    0x2e00..0x2e5e,
    0x2e80..0x2e9a,
    0x2e9b..0x2ef4,
    0x2f00..0x2fd6,
    0x2ff0..0x2ffc,
    0x3001..0x302a,
    0x3030..0x3040,
    0x3041..0x3097,
    0x309b..0x3100,
    0x3105..0x3130,
    0x3131..0x3164,
    0x3165..0x318f,
    0x3190..0x31e4,
    0x31f0..0x321f,
    0x3220..0xa48d,
    0xa490..0xa4c7,
    0xa4d0..0xa62c,
    0xa640..0xa66f,
    0xa673..0xa674,
    0xa67e..0xa69e,
    0xa6a0..0xa6f0,
    0xa6f2..0xa6f8,
    0xa700..0xa7cb,
    0xa7d0..0xa7d2,
    0xa7d3..0xa7d4,
    0xa7d5..0xa7da,
    0xa7f2..0xa802,
    0xa803..0xa806,
    0xa807..0xa80b,
    0xa80c..0xa823,
    0xa828..0xa82c,
    0xa830..0xa83a,
    0xa840..0xa878,
    0xa882..0xa8b4,
    0xa8ce..0xa8da,
    0xa8f2..0xa8ff,
    0xa900..0xa926,
    0xa92e..0xa947,
    0xa95f..0xa97d,
    0xa984..0xa9b3,
    0xa9c1..0xa9ce,
    0xa9cf..0xa9da,
    0xa9de..0xa9e5,
    0xa9e6..0xa9ff,
    0xaa00..0xaa29,
    0xaa40..0xaa43,
    0xaa44..0xaa4c,
    0xaa50..0xaa5a,
    0xaa5c..0xaa7b,
    0xaa7e..0xaab0,
    0xaab1..0xaab2,
    0xaab5..0xaab7,
    0xaab9..0xaabe,
    0xaac0..0xaac1,
    0xaac2..0xaac3,
    0xaadb..0xaaeb,
    0xaaf0..0xaaf5,
    0xab01..0xab07,
    0xab09..0xab0f,
    0xab11..0xab17,
    0xab20..0xab27,
    0xab28..0xab2f,
    0xab30..0xab6c,
    0xab70..0xabe3,
    0xabeb..0xabec,
    0xabf0..0xabfa,
    0xac00..0xd7a4,
    0xd7b0..0xd7c7,
    0xd7cb..0xd7fc,
    0xe000..0xe6ec,
    0xf900..0xfa6e,
    0xfa70..0xfada,
    0xfb00..0xfb07,
    0xfb13..0xfb18,
    0xfb1d..0xfb1e,
    0xfb1f..0xfb37,
    0xfb38..0xfb3d,
    0xfb3e..0xfb3f,
    0xfb40..0xfb42,
    0xfb43..0xfb45,
    0xfb46..0xfbc3,
    0xfbd3..0xfd90,
    0xfd92..0xfdc8,
    0xfdcf..0xfdd0,
    0xfdf0..0xfe00,
    0xfe10..0xfe1a,
    0xfe30..0xfe53,
    0xfe54..0xfe67,
    0xfe68..0xfe6c,
    0xfe70..0xfe75,
    0xfe76..0xfefd,
    0xff01..0xffa0,
    0xffa1..0xffbf,
    0xffc2..0xffc8,
    0xffca..0xffd0,
    0xffd2..0xffd8,
    0xffda..0xffdd,
    0xffe0..0xffe7,
    0xffe8..0xffef,
];
