use crate::data::market::MarketMeta;
use crate::portfolio::error::PortfolioError;
use crate::strategy::signal::Decision;
use chrono::{DateTime, Utc};
use serde::{Deserialize, Serialize};
use uuid::Uuid;

/// Orders are generated by the portfolio and detail work to be done by an execution::handler to
/// execute a trade.
#[derive(Debug, PartialOrd, PartialEq, Serialize, Deserialize)]
pub struct OrderEvent {
    pub event_type: &'static str,
    pub trace_id: Uuid,
    pub timestamp: DateTime<Utc>,
    pub exchange: String,
    pub symbol: String,
    pub market_meta: MarketMeta, // Metadata propagated from source MarketEvent
    pub decision: Decision,      // LONG, CloseLong, SHORT or CloseShort
    pub quantity: f64,           // +ve or -ve Quantity depending on Decision
    pub order_type: OrderType,   // MARKET, LIMIT etc
}

impl Default for OrderEvent {
    fn default() -> Self {
        Self {
            event_type: OrderEvent::EVENT_TYPE,
            trace_id: Uuid::new_v4(),
            timestamp: Utc::now(),
            exchange: String::from("BINANCE"),
            symbol: String::from("ETH-USD"),
            market_meta: MarketMeta::default(),
            decision: Decision::default(),
            quantity: 1.0,
            order_type: OrderType::default(),
        }
    }
}

impl OrderEvent {
    pub const EVENT_TYPE: &'static str = "OrderEvent";

    /// Returns a OrderEventBuilder instance.
    pub fn builder() -> OrderEventBuilder {
        OrderEventBuilder::new()
    }
}

/// Type of order the portfolio wants the execution::handler to place.
#[derive(Debug, Clone, PartialOrd, PartialEq, Eq, Hash, Serialize, Deserialize)]
pub enum OrderType {
    Market,
    Limit,
    Bracket,
}

impl Default for OrderType {
    fn default() -> Self {
        Self::Market
    }
}

/// Builder to construct OrderEvent instances.
#[derive(Debug, Default)]
pub struct OrderEventBuilder {
    pub trace_id: Option<Uuid>,
    pub timestamp: Option<DateTime<Utc>>,
    pub exchange: Option<String>,
    pub symbol: Option<String>,
    pub market_meta: Option<MarketMeta>,
    pub decision: Option<Decision>,
    pub quantity: Option<f64>,
    pub order_type: Option<OrderType>,
}

impl OrderEventBuilder {
    pub fn new() -> Self {
        Self::default()
    }

    pub fn trace_id(self, value: Uuid) -> Self {
        Self {
            trace_id: Some(value),
            ..self
        }
    }

    pub fn timestamp(self, value: DateTime<Utc>) -> Self {
        Self {
            timestamp: Some(value),
            ..self
        }
    }

    pub fn exchange(self, value: String) -> Self {
        Self {
            exchange: Some(value),
            ..self
        }
    }

    pub fn symbol(self, value: String) -> Self {
        Self {
            symbol: Some(value),
            ..self
        }
    }

    pub fn market_meta(self, value: MarketMeta) -> Self {
        Self {
            market_meta: Some(value),
            ..self
        }
    }

    pub fn decision(self, value: Decision) -> Self {
        Self {
            decision: Some(value),
            ..self
        }
    }

    pub fn quantity(self, value: f64) -> Self {
        Self {
            quantity: Some(value),
            ..self
        }
    }

    pub fn order_type(self, value: OrderType) -> Self {
        Self {
            order_type: Some(value),
            ..self
        }
    }

    pub fn build(self) -> Result<OrderEvent, PortfolioError> {
        let trace_id = self.trace_id.ok_or(PortfolioError::BuilderIncomplete)?;
        let timestamp = self.timestamp.ok_or(PortfolioError::BuilderIncomplete)?;
        let exchange = self.exchange.ok_or(PortfolioError::BuilderIncomplete)?;
        let symbol = self.symbol.ok_or(PortfolioError::BuilderIncomplete)?;
        let market_meta = self.market_meta.ok_or(PortfolioError::BuilderIncomplete)?;
        let decision = self.decision.ok_or(PortfolioError::BuilderIncomplete)?;
        let quantity = self.quantity.ok_or(PortfolioError::BuilderIncomplete)?;
        let order_type = self.order_type.ok_or(PortfolioError::BuilderIncomplete)?;

        Ok(OrderEvent {
            event_type: OrderEvent::EVENT_TYPE,
            trace_id,
            timestamp,
            exchange,
            symbol,
            market_meta,
            decision,
            quantity,
            order_type,
        })
    }
}
