use crate::socket::{
    error::SocketError,
    protocol::ProtocolParser,
};
use std::{
    fmt::Debug,
    collections::VecDeque,
    task::{Context, Poll},
    pin::Pin,
    marker::PhantomData
};
use serde::de::DeserializeOwned;
use futures::{Sink, Stream};
use pin_project::pin_project;

/// Contains `ProtocolParser` implementations for transforming communication protocol specific
/// messages into a generic output data structure.
pub mod protocol;

/// Custom `SocketError`s generated by an `ExchangeSocket`.
pub mod error;

/// `Transformer`s are capable of transforming any `Input` into an iterator of
/// `Result<Output, SocketError>`s.
pub trait Transformer<Output> {
    type Input: DeserializeOwned;
    type OutputIter: IntoIterator<Item = Result<Output, SocketError>>;
    fn transform(&mut self, input: Self::Input) -> Self::OutputIter;
}

#[derive(Debug)]
/// Generic event generated by an [`ExchangeSocket`] `Stream`. Contains a monotonically increasing
/// sequence number to support determining event order from the socket.
pub struct Event<Output>
where
    Output: Debug,
{
    pub sequence: u64,
    pub payload: Output,
}

#[derive(Debug)]
/// An `ExchangeSocket` is capable of acting as both a `Stream` and a `Sink` with any communication
/// protocol. It streams exchange messages from the inner socket and transforms them into the
/// desired output data structure.
#[pin_project]
pub struct ExchangeSocket<Protocol, Socket, StreamTransformer, Output>
where
    Protocol: ProtocolParser,
    Socket: Sink<Protocol::Message> + Stream,
    StreamTransformer: Transformer<Output>,
    Output: Debug,
{
    #[pin]
    pub socket: Socket,
    pub sequence: u64,
    pub transformer: StreamTransformer,
    pub buffer: VecDeque<Result<Event<Output>, SocketError>>,
    pub protocol_marker: PhantomData<Protocol>,
}

impl<Protocol, Socket, StreamTransformer, ExchangeMessage, Output> Stream
    for ExchangeSocket<Protocol, Socket, StreamTransformer, Output>
where
    Protocol: ProtocolParser,
    Socket: Sink<Protocol::Message> + Stream<Item = Result<Protocol::Message, Protocol::Error>> + Unpin,
    StreamTransformer: Transformer<Output, Input = ExchangeMessage>,
    ExchangeMessage: DeserializeOwned,
    Output: Debug,
{
    type Item = Result<Event<Output>, SocketError>;

    fn poll_next(mut self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Option<Self::Item>> {
        loop {
            // Flush Self::Item buffer if it is not currently empty
            if let Some(output) = self.buffer.pop_front() {
                return Poll::Ready(Some(output))
            }

            // Poll underlying `Stream` for next `StreamItem` input
            let input = match self.as_mut().project().socket.poll_next(cx) {
                Poll::Ready(Some(input)) => input,
                Poll::Ready(None) => return Poll::Ready(None),
                Poll::Pending => return Poll::Pending,
            };

            // Parse input `StreamItem` into `ExchangeMessage`
            let exchange_message = match Protocol::parse::<ExchangeMessage>(input) {
                // `ProtocolParser` successfully deserialised `ExchangeMessage`
                Some(Ok(exchange_message)) => exchange_message,

                // If `ProtocolParser` returns an Err pass it downstream
                Some(Err(err)) => return Poll::Ready(Some(Err(err))),

                // If `ProtocolParser` returns None it's a safe-to-skip message
                None => return Poll::Pending,
            };

            // Transform `ExchangeMessage` into `Transformer::OutputIter`
            // ie/ IntoIterator<Item = Result<Output, SocketError>>
            self.transformer
                .transform(exchange_message)
                .into_iter()
                .for_each(|output: Result<Output, SocketError>| {

                    // Augment `Output` with monotonically increasing sequence number
                    let event = output
                        .map(|output| {
                            let sequence = self.sequence;
                            self.sequence += 1;
                            Event { sequence, payload: output}
                    });

                    self.buffer.push_back(event)
                });
        }
    }
}

impl<Protocol, Socket, StreamTransformer, Output> Sink<Protocol::Message>
    for ExchangeSocket<Protocol, Socket, StreamTransformer, Output>
where
    Protocol: ProtocolParser,
    Socket: Sink<Protocol::Message> + Stream,
    StreamTransformer: Transformer<Output>,
    Output: Debug,
{
    type Error = SocketError;

    fn poll_ready(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Result<(), Self::Error>> {
        self.project().socket.poll_ready(cx).map_err(|_| SocketError::Sink)
    }

    fn start_send(self: Pin<&mut Self>, item: Protocol::Message) -> Result<(), Self::Error> {
        self.project().socket.start_send(item).map_err(|_| SocketError::Sink)
    }

    fn poll_flush(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Result<(), Self::Error>> {
        self.project().socket.poll_flush(cx).map_err(|_| SocketError::Sink)
    }

    fn poll_close(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Result<(), Self::Error>> {
        self.project().socket.poll_close(cx).map_err(|_| SocketError::Sink)
    }
}

impl<Protocol, Socket, StreamTransformer, Output>
    ExchangeSocket<Protocol, Socket, StreamTransformer, Output>
where
    Protocol: ProtocolParser,
    Socket: Sink<Protocol::Message> + Stream,
    StreamTransformer: Transformer<Output>,
    Output: Debug,
{
    pub fn new(socket: Socket, transformer: StreamTransformer) -> Self {
        Self {
            socket,
            sequence: 0,
            transformer,
            buffer: VecDeque::with_capacity(6),
            protocol_marker: PhantomData::default(),
        }
    }
}