mod barbfile;
mod executor;
mod output;

use jq_rs;
use std::slice::Iter;

use barbfile::BarbFile;
use executor::{Context, Executor};
use output::BarbOutput;

use clap::Parser;

use std::env;
use std::fs;
use std::str::FromStr;

use dotenv::dotenv;

#[derive(Parser, Debug)]
#[clap(version)]
struct Args {
    #[clap(short, long)]
    headers: bool,
    #[clap(short, long)]
    all_headers: bool,
    #[clap(short, long)]
    body: bool,
    #[clap(short, long)]
    raw: bool,
    #[clap(short, long)]
    filter: Option<String>,
    #[clap(short, long)]
    path: Option<String>,
    #[clap(short, long)]
    no_color: bool,
    files: Vec<String>,
}

impl Args {
    pub fn files_iter(&self) -> Iter<String> {
        self.files.iter()
    }

    pub fn jq_filter(&self) -> &Option<String> {
        &self.filter
    }

    pub fn output(&self) -> BarbOutput {
        BarbOutput::new(
            !self.body,
            self.all_headers,
            self.headers || self.all_headers || !self.body,
            !self.headers || self.body,
            self.raw,
            !self.no_color,
        )
    }
}

fn apply_filters(args: &Args, bfile: &BarbFile, body: String) -> Result<String, String> {
    if let Some(filter) = args.jq_filter() {
        return Ok(jq_rs::run(filter.as_str(), body.as_str()).map_err(|x| x.to_string())?);
    } else if let Some(filter) = bfile.filter() {
        return Ok(jq_rs::run(filter.as_str(), body.as_str()).map_err(|x| x.to_string())?);
    }
    Ok(String::from(body))
}

fn run_file(args: &Args, executor: &mut Executor, file_name: &String) -> Result<(), String> {
    let bfile = BarbFile::from_str(
        fs::read_to_string(file_name.as_str())
            .map_err(|_| format!("Failed to read file '{}'", file_name))?
            .as_str(),
    )
    .map_err(|_| format!("Failed to parse file '{}'", file_name))?;
    let output = args.output();
    let response = executor.execute(&bfile, &output)?;

    output.status(response.status(), response.status_text());
    for header_name in response.headers_names() {
        output.resp_hdr(
            header_name.to_string(),
            response.header(header_name.as_str()).unwrap(),
        );
    }
    output.end_resp_hdr();

    output.body(apply_filters(
        args,
        &bfile,
        response.into_string().unwrap(),
    )?);

    Ok(())
}

fn main() {
    let args = Args::parse();
    dotenv().ok();
    let mut executor = Executor::new(Context::new(env::vars()));

    for file in args.files_iter() {
        match run_file(&args, &mut executor, file) {
            Ok(()) => (),
            Err(err) => println!("{}", err),
        }
    }
}
