use crate::barbfile::BarbFile;
use crate::output::BarbOutput;

use std::collections::HashMap;
use ureq;
use ureq::Error as UreqError;

pub struct Context {
    vars: HashMap<String, String>,
}

impl Context {
    pub fn new<I>(vars: I) -> Context
    where
        I: Iterator<Item = (String, String)>,
    {
        let mut toto = HashMap::new();
        toto.extend(vars);

        Context { vars: toto }
    }

    #[cfg(test)]
    pub fn empty() -> Context {
        Context {
            vars: HashMap::new(),
        }
    }

    // Preserved for future reference
    // pub fn get_var(&self, name: String) -> Option<String> {
    //     self.vars
    //         .get(&name)
    //         .map(|val| val.clone())
    //         .or_else(|| env::var(name).ok())
    // }

    fn key_str(&self, key: &String) -> String {
        format!("{{{}}}", key)
    }

    pub fn substitute(&self, string: &String) -> String {
        let mut buffer = string.clone();
        for (key, val) in self.vars.iter() {
            buffer = buffer.replace(self.key_str(key).as_str(), val);
        }

        buffer
    }
}

pub struct Executor {
    context: Context,
}

impl Executor {
    pub fn new(context: Context) -> Executor {
        Executor { context }
    }

    fn make_req(&self, bfile: &BarbFile, output: &BarbOutput) -> ureq::Request {
        output.req(
            bfile.method_as_string(),
            self.context.substitute(&bfile.url()),
        );
        let mut req = ureq::request(
            bfile.method_as_string().as_str(),
            self.context.substitute(&bfile.url()).as_str(),
        );

        for header in bfile.headers() {
            let hdr_val = self.context.substitute(header.value());
            req = req.set(header.name(), hdr_val.as_str());
            output.req_hdr(header.name().to_string(), hdr_val);
        }

        output.end_req();

        req
    }

    fn run(&self, bfile: &BarbFile, req: ureq::Request) -> Result<ureq::Response, String> {
        let resp = match bfile.method().takes_body() {
            true => match bfile.body() {
                Some(body) => req.send_string(body.as_str()),
                None => req.call(),
            },
            false => req.call(),
        };

        match resp {
            Ok(resp) => Ok(resp),
            Err(UreqError::Status(_, resp)) => Ok(resp),
            Err(UreqError::Transport(transp)) => Err(String::from(transp.to_string())),
        }
    }

    pub fn execute(
        &mut self,
        bfile: &BarbFile,
        output: &BarbOutput,
    ) -> Result<ureq::Response, String> {
        self.run(bfile, self.make_req(&bfile, output))
    }
}

// TODO: tests
#[cfg(test)]
mod tests {
    use super::*;
    use std::str::FromStr;

    #[test]
    fn test_context_key_str() {
        let ctx = Context::empty();
        assert_eq!(ctx.key_str(&String::from("foo")), String::from("{foo}"));
    }

    #[test]
    fn test_context_substitute() {
        let vars: Vec<(String, String)> = vec![
            (String::from("foo"), String::from("bar")),
            (String::from("bar"), String::from("baz")),
        ];
        let ctx = Context::new(vars.into_iter());
        assert_eq!(
            ctx.substitute(&String::from("blah blah {foo} blah")),
            String::from("blah blah bar blah")
        );
        assert_eq!(
            ctx.substitute(&String::from("blah {foo} {bar} blah")),
            String::from("blah bar baz blah")
        );
    }

    #[test]
    fn test_make_req_simple_get() {
        let executor = Executor::new(Context::empty());
        let bfile = BarbFile::from_str("#GET^http://foo.bar\n\n").unwrap();
        let output = BarbOutput::new(false, false, false, false, false, false);
        let req = executor.make_req(&bfile, &output);

        assert_eq!(req.url(), "http://foo.bar");
        assert_eq!(req.method(), "GET");
    }

    #[test]
    fn test_make_req_simple_headers() {
        let executor = Executor::new(Context::empty());
        let bfile = BarbFile::from_str("#GET^http://foo.bar\n#Foo: Bar\n#Bar: Baz\n\n").unwrap();
        let output = BarbOutput::new(false, false, false, false, false, false);
        let req = executor.make_req(&bfile, &output);

        assert_eq!(
            req.header_names(),
            vec![String::from("foo"), String::from("bar")]
        );
        assert_eq!(req.header("foo"), Some("Bar"));
        assert_eq!(req.header("bar"), Some("Baz"));
    }
}
