mod barbfile;
mod executor;

use jsonformat::{format_json, Indentation};
use std::slice::Iter;

use barbfile::BarbFile;
use executor::{Context, Executor};

use clap::Parser;

use std::env;
use std::fs;
use std::str::FromStr;

#[derive(Parser, Debug)]
#[clap(version)]
struct Args {
    #[clap(short, long)]
    headers: bool,
    #[clap(short, long)]
    all_headers: bool,
    #[clap(short, long)]
    body: bool,
    #[clap(short, long)]
    raw: bool,
    files: Vec<String>,
}

impl Args {
    pub fn print_headers(&self) -> bool {
        self.headers || self.all_headers || !self.body
    }

    pub fn print_body(&self) -> bool {
        !self.headers || self.body
    }

    pub fn raw_body(&self) -> bool {
        self.raw
    }

    pub fn req_headers(&self) -> bool {
        self.all_headers
    }

    pub fn files_iter(&self) -> Iter<String>
    {
        self.files.iter()
    }
}

fn run_file(args: &Args, executor: &mut Executor, file_name: &String) {
    let bfile = BarbFile::from_str(
        fs::read_to_string(file_name.as_str())
            .expect("Failed to read file")
            .as_str(),
    )
    .expect("Failed to parse file");
    let response = executor.execute(bfile, args.req_headers()).unwrap();

    if args.print_headers() {
        println!("{} {}", response.status(), response.status_text());
        for header_name in response.headers_names() {
            println!(
                "{}: {}",
                header_name,
                response.header(header_name.as_str()).unwrap()
            );
        }
    }

    if args.print_body() {
        println!(
            "{}",
            match args.raw_body() {
                true => String::from(response.into_string().unwrap().as_str()),
                false => format_json(
                    response.into_string().unwrap().as_str(),
                    Indentation::Default
                ),
            }
        );
    }
}

fn main() {
    let args = Args::parse();

    let mut executor = Executor::new(Context::new(env::vars()));

    for file in args.files_iter() {
        run_file(&args, &mut executor, &file);
    }
}
