// Copyright 2021 Stichting Organism
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

//! Test cases based on reference python code and can be found in the github bacteria_test repo

use crate::{
    internal::SPONGE_BLOCK_SIZE,
    strobe::Strobe128,
};

#[test]
fn test_init_128() {
    let s = Strobe128::new(b"");
    let initial_st = s.state.0;
    let expected_st: &[u8; 8 * SPONGE_BLOCK_SIZE] = &[
        0x9c, 0x7f, 0x16, 0x8f, 0xf8, 0xfd, 0x55, 0xda, 0x2a, 0xa7, 0x3c, 0x23, 0x55, 0x65, 0x35,
        0x63, 0xdc, 0x0c, 0x47, 0x5c, 0x55, 0x15, 0x26, 0xf6, 0x73, 0x3b, 0xea, 0x22, 0xf1, 0x6c,
        0xb5, 0x7c, 0xd3, 0x1f, 0x68, 0x2e, 0x66, 0x0e, 0xe9, 0x12, 0x82, 0x4a, 0x77, 0x22, 0x01,
        0xee, 0x13, 0x94, 0x22, 0x6f, 0x4a, 0xfc, 0xb6, 0x2d, 0x33, 0x12, 0x93, 0xcc, 0x92, 0xe8,
        0xa6, 0x24, 0xac, 0xf6, 0xe1, 0xb6, 0x00, 0x95, 0xe3, 0x22, 0xbb, 0xfb, 0xc8, 0x45, 0xe5,
        0xb2, 0x69, 0x95, 0xfe, 0x7d, 0x7c, 0x84, 0x13, 0x74, 0xd1, 0xff, 0x58, 0x98, 0xc9, 0x2e,
        0xe0, 0x63, 0x6b, 0x06, 0x72, 0x73, 0x21, 0xc9, 0x2a, 0x60, 0x39, 0x07, 0x03, 0x53, 0x49,
        0xcc, 0xbb, 0x1b, 0x92, 0xb7, 0xb0, 0x05, 0x7e, 0x8f, 0xa8, 0x7f, 0xce, 0xbc, 0x7e, 0x88,
        0x65, 0x6f, 0xcb, 0x45, 0xae, 0x04, 0xbc, 0x34, 0xca, 0xbe, 0xae, 0xbe, 0x79, 0xd9, 0x17,
        0x50, 0xc0, 0xe8, 0xbf, 0x13, 0xb9, 0x66, 0x50, 0x4d, 0x13, 0x43, 0x59, 0x72, 0x65, 0xdd,
        0x88, 0x65, 0xad, 0xf9, 0x14, 0x09, 0xcc, 0x9b, 0x20, 0xd5, 0xf4, 0x74, 0x44, 0x04, 0x1f,
        0x97, 0xb6, 0x99, 0xdd, 0xfb, 0xde, 0xe9, 0x1e, 0xa8, 0x7b, 0xd0, 0x9b, 0xf8, 0xb0, 0x2d,
        0xa7, 0x5a, 0x96, 0xe9, 0x47, 0xf0, 0x7f, 0x5b, 0x65, 0xbb, 0x4e, 0x6e, 0xfe, 0xfa, 0xa1,
        0x6a, 0xbf, 0xd9, 0xfb, 0xf6,
    ];

    assert_eq!(&initial_st[..], &expected_st[..]);
}


#[test]
fn test_seq() {
    let mut s = Strobe128::new(b"seqtest");

    let mut buf = [0u8; 10];
    s.prf(&mut buf[..], false);

    s.ad(b"Hello", false);

    let mut buf = b"World".to_vec();
    s.send_enc(buf.as_mut_slice(), false);

    s.send_clr(b"foo", false);
    s.ratchet(32, false);
    s.recv_clr(b"bar", false);

    let mut buf = b"baz".to_vec();
    s.recv_enc(buf.as_mut_slice(), false);

    for i in 0..100 {
        let mut buf = vec![b'X'; i];
        s.send_enc(buf.as_mut_slice(), false);
    }

    let mut buf = [0u8; 123];
    s.prf(&mut buf[..], false);

    let mut buf = [0u8; 16];
    s.send_mac(&mut buf[..], false);

    let final_st = s.state.0;
    let expected_st = [0x0c, 0x21, 0x96, 0x3b, 0x9a, 0xfb, 0xdf, 0xdc, 0x72, 0x3a, 0x78, 0xbf, 0x84, 0x95, 0x6f, 0x15, 0x99, 0x30, 0x66, 0x8b, 0x29, 0x1c, 0xca, 0x03, 0x60, 0x21, 0x84, 0xf5, 0xc8, 0x10, 0xc7, 0xbf, 0x52, 0x08, 0x28, 0x59, 0x0c, 0x03, 0xde, 0x5c, 0xfa, 0x82, 0xbc, 0xcd, 0xb9, 0x09, 0x16, 0xab, 0x98, 0x4d, 0xbf, 0x14, 0x28, 0x0e, 0x59, 0xf1, 0x50, 0x98, 0x85, 0xb5, 0xe7, 0x3c, 0x6b, 0x0c, 0x54, 0xe8, 0x3d, 0x1c, 0x67, 0x78, 0xba, 0x96, 0x53, 0xdb, 0xf2, 0x9f, 0xbe, 0x6f, 0xa8, 0x38, 0x23, 0x70, 0xfd, 0x97, 0xb8, 0x78, 0xdd, 0x90, 0xff, 0x46, 0x79, 0x29, 0xe0, 0x6f, 0x47, 0x2e, 0x6a, 0x1d, 0x08, 0xba, 0xb2, 0x66, 0x18, 0x90, 0x1b, 0x05, 0xa9, 0x77, 0xd6, 0x2f, 0x87, 0x24, 0xe8, 0x99, 0x6c, 0x86, 0x2c, 0xef, 0x31, 0x8c, 0x3b, 0x5e, 0x01, 0xe8, 0x62, 0x8b, 0xe2, 0x3b, 0x5c, 0x47, 0x23, 0x9b, 0xb7, 0xbb, 0xd4, 0x47, 0x89, 0xae, 0x7f, 0x56, 0xb7, 0xae, 0xff, 0x89, 0x48, 0x7d, 0x7b, 0x84, 0xa6, 0xed, 0x36, 0xf4, 0x1b, 0x19, 0xb5, 0x96, 0x15, 0x13, 0x5d, 0x42, 0xb9, 0x97, 0xbb, 0x49, 0x71, 0x6c, 0x65, 0x8f, 0xfb, 0x3f, 0xf9, 0xde, 0x80, 0xc3, 0xcf, 0xcd, 0x00, 0xd3, 0x75, 0x42, 0xde, 0xab, 0xe8, 0x54, 0x5f, 0xa6, 0x2f, 0xc1, 0xb6, 0x0b, 0xf4, 0xc6, 0x07, 0x2e, 0x62, 0x39, 0x6d, 0xda, 0x93, 0x6d];

    assert_eq!(&final_st[..], &expected_st[..]);
}


#[test]
fn test_metadata() {
    // We will accumulate output over 3 operations and 3 meta-operations
    let mut s = Strobe128::new(b"metadatatest");
    let mut output = Vec::new();

    let buf = b"meta1";
    s.meta_send_clr(buf, false);
    output.extend_from_slice(buf);

    // This does not output anything
    s.key(b"key", false);

    let mut buf = [0u8; 10];
    s.meta_prf(&mut buf, false);
    output.extend_from_slice(&buf[..]);

    // We don't have to re-zero the buffer. Our internal special-casing for PRF does this for us
    s.prf(&mut buf, false);
    output.extend_from_slice(&buf[..]);

    let buf = b"meta3";
    s.meta_send_clr(buf, false);
    output.extend(buf);

    let mut buf = b"pt".to_vec();
    s.send_enc(buf.as_mut_slice(), false);
    output.extend(buf);

    let expected_output = [0x6d, 0x65, 0x74, 0x61, 0x31, 0x28, 0xf0, 0xd6, 0xd0, 0x1b, 0x7c, 0x7a, 0x26, 0x42, 0x92, 0x7c, 0x49, 0x3b, 0x06, 0xd9, 0x20, 0xd6, 0x14, 0xed, 0xa0, 0x6d, 0x65, 0x74, 0x61, 0x33, 0xdf, 0xa9];
    let expected_st = [0xdf, 0xa9, 0x45, 0xbf, 0x36, 0x7d, 0xc6, 0xa4, 0x0d, 0xbf, 0xc8, 0x97, 0x97, 0x8c, 0x36, 0xf9, 0x70, 0x73, 0x8a, 0xc8, 0x2e, 0xba, 0x6b, 0xc7, 0xba, 0xd5, 0xe0, 0x4d, 0x11, 0x30, 0xd3, 0x91, 0xcf, 0xa9, 0x23, 0x7d, 0x4a, 0xc9, 0x98, 0x17, 0x62, 0x34, 0x2e, 0x64, 0xc7, 0xa6, 0x34, 0x50, 0xab, 0xdf, 0xbe, 0x18, 0xbf, 0x52, 0xa1, 0x70, 0x00, 0x1c, 0x20, 0x92, 0xc7, 0x7e, 0x91, 0x8c, 0xf5, 0x14, 0xcb, 0xdc, 0x8e, 0xee, 0xdb, 0x11, 0xa2, 0xd1, 0xff, 0x21, 0x33, 0x6d, 0x64, 0x43, 0x2a, 0x85, 0xe6, 0xe3, 0x30, 0x81, 0x54, 0xc0, 0x97, 0x80, 0x10, 0x3b, 0x42, 0x31, 0xdf, 0xff, 0x5b, 0x09, 0x6e, 0x3f, 0x90, 0x3a, 0x56, 0x5b, 0xfb, 0x11, 0x2b, 0xf4, 0xe8, 0xc8, 0x87, 0xfe, 0xe4, 0x15, 0x1b, 0x0b, 0xfe, 0xab, 0x7c, 0x98, 0x96, 0x8f, 0xa7, 0x02, 0x9b, 0xa1, 0x98, 0x73, 0xa6, 0xf5, 0x5f, 0xa0, 0x88, 0x8a, 0x07, 0x49, 0x86, 0xfc, 0xd6, 0x21, 0x4a, 0x28, 0xc7, 0xec, 0x4e, 0xb1, 0x24, 0xc1, 0x8f, 0x00, 0xcb, 0xa9, 0x71, 0x1b, 0x57, 0x08, 0x2d, 0x0c, 0x7d, 0xe6, 0xea, 0x4f, 0x6a, 0x8b, 0x51, 0x32, 0x6b, 0xdf, 0x49, 0x59, 0x8a, 0x33, 0x92, 0xd6, 0xf5, 0x8f, 0xb4, 0xe0, 0x2c, 0xd6, 0x02, 0xc2, 0x57, 0x40, 0x9a, 0x2a, 0x40, 0x5d, 0xf2, 0xf5, 0xad, 0x78, 0x3c, 0xde, 0x5d, 0x06, 0x35, 0x8b, 0x59, 0xf2];

    assert_eq!(&output, &expected_output);
    assert_eq!(&s.state.0[..], &expected_st[..]);
}


#[test]
fn test_long_inputs() {
    let mut s = Strobe128::new(b"bigtest");
    const BIG_N: usize = 9823;
    let big_data = [0x34; BIG_N];

    s.meta_ad(&big_data[..], false);
    s.ad(&big_data[..], false);
    s.meta_key(&big_data[..], false);
    s.key(&big_data[..], false);
    s.meta_send_clr(&big_data[..], false);
    s.send_clr(&big_data[..], false);
    s.meta_recv_clr(&big_data[..], false);
    s.recv_clr(&big_data[..], false);

    s.meta_send_enc(big_data.to_vec().as_mut_slice(), false);
    s.send_enc(big_data.to_vec().as_mut_slice(), false);
    s.meta_recv_enc(big_data.to_vec().as_mut_slice(), false);
    s.recv_enc(big_data.to_vec().as_mut_slice(), false);
    let _ = s.meta_recv_mac(big_data.to_vec().as_mut_slice(), false);
    let _ = s.recv_mac(big_data.to_vec().as_mut_slice(), false);

    let mut big_buf = [0u8; BIG_N];

    s.meta_ratchet(BIG_N, false);
    s.ratchet(BIG_N, false);
    s.meta_prf(&mut big_buf, false);
    s.prf(&mut big_buf, false);
    s.meta_send_mac(&mut big_buf, false);
    s.send_mac(&mut big_buf, false);

    let expected_st = [0x44, 0x1b, 0x94, 0x3d, 0x24, 0x24, 0x45, 0xe0, 0x41, 0x2a, 0x7e, 0x0b, 0x20, 0x04, 0x86, 0x90, 0xdf, 0xaf, 0x92, 0xda, 0x72, 0x70, 0x30, 0x77, 0x2c, 0x2b, 0x59, 0x0b, 0x26, 0xfa, 0xa1, 0xf1, 0x54, 0x6a, 0x8c, 0xd4, 0x62, 0x0f, 0x61, 0xbf, 0xfd, 0x0f, 0x78, 0x98, 0x0f, 0x92, 0x1a, 0xe4, 0xe2, 0x04, 0x4c, 0x4b, 0x61, 0x07, 0x58, 0x59, 0x35, 0x85, 0xed, 0x0a, 0x24, 0x1d, 0x84, 0x80, 0xc1, 0x19, 0xbf, 0xf7, 0xbb, 0xc2, 0x5a, 0xe5, 0x93, 0x63, 0x44, 0x9e, 0xb4, 0x90, 0x96, 0xb6, 0x3a, 0x33, 0x1b, 0x5e, 0x17, 0x99, 0xaa, 0x49, 0x80, 0xd1, 0x59, 0x8e, 0x99, 0xe7, 0xc6, 0xdf, 0xcd, 0x52, 0xba, 0x0d, 0x17, 0xb7, 0x86, 0xa4, 0x4e, 0x56, 0x47, 0xbe, 0xe5, 0x92, 0xef, 0x2c, 0xb4, 0x23, 0x27, 0xde, 0x6e, 0x8d, 0xea, 0xa7, 0x46, 0x5f, 0x1d, 0x01, 0xcf, 0xaf, 0xe0, 0x9b, 0xf8, 0xdc, 0x04, 0xc2, 0x47, 0x4f, 0x31, 0x85, 0x6d, 0x61, 0xa4, 0x2b, 0x14, 0xb7, 0x50, 0xa2, 0x4c, 0x0f, 0x7c, 0x4e, 0xde, 0x17, 0xde, 0x24, 0x1c, 0xd6, 0x9b, 0x2f, 0xb8, 0x96, 0xe8, 0xf3, 0x26, 0x2d, 0x2c, 0xe4, 0xa6, 0x14, 0xd0, 0xe2, 0x60, 0x97, 0x34, 0x49, 0xbb, 0x3a, 0x4d, 0x46, 0x4f, 0x20, 0x5f, 0x30, 0x32, 0x5e, 0xa0, 0x42, 0x8e, 0x12, 0xb6, 0x40, 0xd9, 0x56, 0x1c, 0x29, 0x7a, 0x2c, 0xd3, 0x91, 0xba, 0xe4, 0x90, 0x0b];

    assert_eq!(&s.state.0[..], &expected_st[..]);
}

// Test that streaming in data using the `more` flag works as expected
#[test]
fn test_streaming() {
    // Compute a few things without breaking up their inputs
    let one_shot_st: Vec<u8> = {
        let mut s = Strobe128::new(b"streamingtest");
        s.ad(b"mynonce", false);

        let mut buf = b"hello there".to_vec();
        s.recv_enc(buf.as_mut_slice(), false);

        let mut buf = [0u8; 16];
        s.send_mac(&mut buf[..], false);

        s.state.0.to_vec()
    };
    // Now do the same thing but stream the inputs
    let streamed_st: Vec<u8> = {
        let mut s = Strobe128::new(b"streamingtest");
        s.ad(b"my", false);
        s.ad(b"nonce", true);

        let mut buf = b"hello".to_vec();
        s.recv_enc(buf.as_mut_slice(), false);

        let mut buf = b" there".to_vec();
        s.recv_enc(buf.as_mut_slice(), true);

        let mut buf = [0u8; 10];
        s.send_mac(&mut buf[..], false);
        let mut buf = [0u8; 6];
        s.send_mac(&mut buf[..], true);
        s.state.0.to_vec()
    };

    assert_eq!(one_shot_st, streamed_st);
}

// Test that decrypt(encrypt(msg)) == msg
#[test]
fn test_enc_correctness() {
    let orig_msg = b"Hello there";
    let mut tx = Strobe128::new(b"enccorrectnesstest");
    let mut rx = Strobe128::new(b"enccorrectnesstest");

    tx.key(b"the-combination-on-my-luggage", false);
    rx.key(b"the-combination-on-my-luggage", false);

    // Encrypt and decrypt the original message
    let mut buf = orig_msg.to_vec();
    tx.send_enc(buf.as_mut_slice(), false);
    rx.recv_enc(buf.as_mut_slice(), false);

    assert_eq!(orig_msg, buf.as_slice());
}

// Test that recv_mac(send_mac()) doesn't error, and recv_mac(otherstuff) does error
#[test]
fn test_mac_correctness() {
    let mut tx = Strobe128::new(b"maccorrectnesstest");
    let mut rx = Strobe128::new(b"maccorrectnesstest");

    // Just do some stuff with the state

    tx.key(b"secretsauce", false);
    let mut msg = b"attack at dawn".to_vec();
    tx.send_enc(msg.as_mut_slice(), false);

    let mut mac = [0u8; 16];
    tx.send_mac(&mut mac[..], false);

    rx.key(b"secretsauce", false);
    rx.recv_enc(&mut msg[..], false);

    // Test that valid MACs are accepted
    let mut rx_copy = rx.clone();
    let good_res = rx_copy.recv_mac(&mut mac[..], false);
    assert!(good_res.is_ok());

    // Test that invalid MACs are rejected
    let mut bad_mac = {
        let mut tmp = mac.to_vec();
        tmp.push(0);
        tmp
    };
    let bad_res = rx.recv_mac(&mut bad_mac[..], false);
    assert!(bad_res.is_err());
}

#[test]
fn test_aead_correctness() {
    let orig_msg = b"Hello there";
    let ad = b"metadata";
    let mut tx = Strobe128::new(b"enccorrectnesstest");
    let mut rx = Strobe128::new(b"enccorrectnesstest");

    tx.key(b"the-combination-on-my-luggage", false);
    rx.key(b"the-combination-on-my-luggage", false);

    // Encrypt and decrypt the original message
    let mut buf = orig_msg.to_vec();
    let capsule = tx.send_aead(buf.as_mut_slice(), ad);
    let capsule_back = rx.recv_aead(&capsule, ad).unwrap();
    assert_eq!(orig_msg, capsule_back.as_slice());
}