use crate::clients::{ServiceType, StorageAccountClient};
use crate::table::requests::*;
use azure_core::errors::AzureError;
use bytes::Bytes;
use http::method::Method;
use http::request::{Builder, Request};
use std::sync::Arc;

pub trait AsTableServiceClient {
    fn as_table_service_client(&self) -> Arc<TableServiceClient>;
}

impl AsTableServiceClient for Arc<StorageAccountClient> {
    fn as_table_service_client(&self) -> Arc<TableServiceClient> {
        TableServiceClient::new(self.clone())
    }
}

#[derive(Debug, Clone)]
pub struct TableServiceClient {
    storage_account_client: Arc<StorageAccountClient>,
}

impl TableServiceClient {
    pub(crate) fn new(storage_account_client: Arc<StorageAccountClient>) -> Arc<Self> {
        Arc::new(Self {
            storage_account_client,
        })
    }

    pub(crate) fn storage_account_client(&self) -> &StorageAccountClient {
        self.storage_account_client.as_ref()
    }

    pub fn query_tables(&self) -> QueryTablesBuilder {
        QueryTablesBuilder::new(self)
    }

    pub(crate) fn prepare_request<'a>(
        &self,
        url: &str,
        method: &Method,
        http_header_adder: &dyn Fn(Builder) -> Builder,
        request_body: Option<Bytes>,
    ) -> Result<(Request<Bytes>, url::Url), AzureError> {
        self.storage_account_client.prepare_request(
            url,
            method,
            http_header_adder,
            ServiceType::Table,
            request_body,
        )
    }
}
