use indexmap::IndexMap;
use std::rc::Rc;

use crate::binaryxml::{
    BinaryXmlDocument, XmlCdata, XmlNodeType, XmlStartElement, XmlStartNameSpace,
};
use crate::stringpool::StringPool;
use crate::ParseError;

///Struct representing a parsed XML document.
#[derive(Debug)]
pub struct XmlDocument {
    root: Option<Node>,
}

impl XmlDocument {
    pub(crate) fn new(binaryxml: BinaryXmlDocument) -> Result<Self, ParseError> {
        let string_pool = binaryxml.string_pool;
        let resource_map = binaryxml.resource_map;

        let mut namespaces = IndexMap::new();

        let mut element_tracker: Vec<Element> = Vec::new();
        for node in binaryxml.elements {
            match node.element {
                XmlNodeType::XmlStartNameSpace(e) => {
                    let (uri, prefix) = Self::process_start_namespace(&e, &string_pool)?;
                    namespaces.insert(uri.clone(), prefix.clone());
                }
                XmlNodeType::XmlEndNameSpace(_) => {}
                XmlNodeType::XmlStartElement(e) => {
                    element_tracker.push(Self::process_start_element(
                        &e,
                        &string_pool,
                        &namespaces,
                        &resource_map.resource_ids,
                    )?);
                }
                XmlNodeType::XmlEndElement(_) => {
                    let e = element_tracker.pop().unwrap();

                    if element_tracker.is_empty() {
                        return Ok(XmlDocument {
                            root: Some(Node::Element(e)),
                        });
                    }

                    element_tracker
                        .last_mut()
                        .unwrap()
                        .insert_children(Node::Element(e));
                }
                XmlNodeType::XmlCdata(e) => {
                    let cdata = Self::process_cdata(&e, &string_pool)?;
                    element_tracker
                        .last_mut()
                        .unwrap()
                        .insert_children(Node::Cdata(cdata))
                }
            };
        }

        Ok(Self { root: None })
    }

    ///Returns the root [Element] of the XML document.
    pub fn get_root(&self) -> &Option<Node> {
        &self.root
    }

    fn process_cdata(e: &XmlCdata, string_pool: &StringPool) -> Result<Cdata, ParseError> {
        Ok(Cdata {
            data: string_pool
                .get(usize::try_from(e.data).unwrap())
                .ok_or(ParseError::StringNotFound(e.data))?
                .to_string(),
        })
    }

    fn process_start_namespace(
        e: &XmlStartNameSpace,
        string_pool: &StringPool,
    ) -> Result<(Rc<String>, Rc<String>), ParseError> {
        let uri = string_pool
            .get(usize::try_from(e.uri).unwrap())
            .ok_or(ParseError::StringNotFound(e.uri))?;
        let prefix = string_pool
            .get(usize::try_from(e.prefix).unwrap())
            .ok_or(ParseError::StringNotFound(e.prefix))?;

        Ok((uri, prefix))
    }

    fn process_start_element(
        e: &XmlStartElement,
        string_pool: &StringPool,
        namespaces: &IndexMap<Rc<String>, Rc<String>>,
        resource_map: &[u32],
    ) -> Result<Element, ParseError> {
        let ns = string_pool.get(usize::try_from(e.attr_ext.ns).unwrap());
        assert_eq!(ns, None);

        let name = string_pool
            .get(usize::try_from(e.attr_ext.name).unwrap())
            .ok_or(ParseError::StringNotFound(e.attr_ext.name))?;
        let name = (*name).clone();

        let mut attributes: IndexMap<String, String> = IndexMap::new();
        for attr in &e.attributes {
            let ns = string_pool.get(usize::try_from(attr.ns).unwrap());
            let name = string_pool
                .get(usize::try_from(attr.name).unwrap())
                .ok_or(ParseError::StringNotFound(attr.name))?;
            let value = attr.typed_value.get_value(string_pool);

            let mut final_name = String::new();
            if !name.is_empty() {
                if let Some(n) = ns {
                    let ns_prefix = namespaces
                        .get(&n)
                        .ok_or_else(|| ParseError::NamespaceNotFound(n.to_string()))?;
                    final_name.push_str(ns_prefix);
                    final_name.push(':');
                }
                final_name.push_str(&name);
            } else {
                let resource_id = resource_map
                    .get(usize::try_from(attr.name).unwrap())
                    .ok_or(ParseError::ResourceIdNotFound(attr.name))?;
                let resource_str = get_resource_string(*resource_id)
                    .ok_or(ParseError::UnknownResourceString(*resource_id))?;
                final_name.push_str(&resource_str);
            }

            attributes.insert(final_name, value.to_string());
        }

        Ok(Element {
            attributes,
            tag: name,
            children: Vec::new(),
        })
    }
}

///Enum representing possible nodes within the parsed XML document.
#[derive(Debug)]
pub enum Node {
    Element(Element),
    Cdata(Cdata),
}

///Struct representing an element within the parsed XML document.
#[derive(Debug)]
pub struct Element {
    attributes: IndexMap<String, String>,
    tag: String,
    children: Vec<Node>,
}

impl Element {
    ///Returns a map of attributes associated with the element.
    pub fn get_attributes(&self) -> &IndexMap<String, String> {
        &self.attributes
    }

    ///Returns the element tag.
    pub fn get_tag(&self) -> &str {
        &self.tag
    }

    ///Returns a list of child nodes.
    pub fn get_children(&self) -> &Vec<Node> {
        &self.children
    }

    fn insert_children(&mut self, child: Node) {
        self.children.push(child);
    }
}

///Struct representing a Cdata element within the parsed XML document.
#[derive(Debug)]
pub struct Cdata {
    data: String,
}

impl Cdata {
    pub fn get_data(&self) -> &str {
        &self.data
    }
}

// Logic borrowed from:
// https://github.com/ytsutano/axmldec/blob/master/lib/jitana/util/axml_parser.cpp#L504
fn get_resource_string(resource_id: u32) -> Option<String> {
    const RESOURCE_STRINGS: &[&str] = &[
        "theme",
        "label",
        "icon",
        "name",
        "manageSpaceActivity",
        "allowClearUserData",
        "permission",
        "readPermission",
        "writePermission",
        "protectionLevel",
        "permissionGroup",
        "sharedUserId",
        "hasCode",
        "persistent",
        "enabled",
        "debuggable",
        "exported",
        "process",
        "taskAffinity",
        "multiprocess",
        "finishOnTaskLaunch",
        "clearTaskOnLaunch",
        "stateNotNeeded",
        "excludeFromRecents",
        "authorities",
        "syncable",
        "initOrder",
        "grantUriPermissions",
        "priority",
        "launchMode",
        "screenOrientation",
        "configChanges",
        "description",
        "targetPackage",
        "handleProfiling",
        "functionalTest",
        "value",
        "resource",
        "mimeType",
        "scheme",
        "host",
        "port",
        "path",
        "pathPrefix",
        "pathPattern",
        "action",
        "data",
        "targetClass",
        "colorForeground",
        "colorBackground",
        "backgroundDimAmount",
        "disabledAlpha",
        "textAppearance",
        "textAppearanceInverse",
        "textColorPrimary",
        "textColorPrimaryDisableOnly",
        "textColorSecondary",
        "textColorPrimaryInverse",
        "textColorSecondaryInverse",
        "textColorPrimaryNoDisable",
        "textColorSecondaryNoDisable",
        "textColorPrimaryInverseNoDisable",
        "textColorSecondaryInverseNoDisable",
        "textColorHintInverse",
        "textAppearanceLarge",
        "textAppearanceMedium",
        "textAppearanceSmall",
        "textAppearanceLargeInverse",
        "textAppearanceMediumInverse",
        "textAppearanceSmallInverse",
        "textCheckMark",
        "textCheckMarkInverse",
        "buttonStyle",
        "buttonStyleSmall",
        "buttonStyleInset",
        "buttonStyleToggle",
        "galleryItemBackground",
        "listPreferredItemHeight",
        "expandableListPreferredItemPaddingLeft",
        "expandableListPreferredChildPaddingLeft",
        "expandableListPreferredItemIndicatorLeft",
        "expandableListPreferredItemIndicatorRight",
        "expandableListPreferredChildIndicatorLeft",
        "expandableListPreferredChildIndicatorRight",
        "windowBackground",
        "windowFrame",
        "windowNoTitle",
        "windowIsFloating",
        "windowIsTranslucent",
        "windowContentOverlay",
        "windowTitleSize",
        "windowTitleStyle",
        "windowTitleBackgroundStyle",
        "alertDialogStyle",
        "panelBackground",
        "panelFullBackground",
        "panelColorForeground",
        "panelColorBackground",
        "panelTextAppearance",
        "scrollbarSize",
        "scrollbarThumbHorizontal",
        "scrollbarThumbVertical",
        "scrollbarTrackHorizontal",
        "scrollbarTrackVertical",
        "scrollbarAlwaysDrawHorizontalTrack",
        "scrollbarAlwaysDrawVerticalTrack",
        "absListViewStyle",
        "autoCompleteTextViewStyle",
        "checkboxStyle",
        "dropDownListViewStyle",
        "editTextStyle",
        "expandableListViewStyle",
        "galleryStyle",
        "gridViewStyle",
        "imageButtonStyle",
        "imageWellStyle",
        "listViewStyle",
        "listViewWhiteStyle",
        "popupWindowStyle",
        "progressBarStyle",
        "progressBarStyleHorizontal",
        "progressBarStyleSmall",
        "progressBarStyleLarge",
        "seekBarStyle",
        "ratingBarStyle",
        "ratingBarStyleSmall",
        "radioButtonStyle",
        "scrollbarStyle",
        "scrollViewStyle",
        "spinnerStyle",
        "starStyle",
        "tabWidgetStyle",
        "textViewStyle",
        "webViewStyle",
        "dropDownItemStyle",
        "spinnerDropDownItemStyle",
        "dropDownHintAppearance",
        "spinnerItemStyle",
        "mapViewStyle",
        "preferenceScreenStyle",
        "preferenceCategoryStyle",
        "preferenceInformationStyle",
        "preferenceStyle",
        "checkBoxPreferenceStyle",
        "yesNoPreferenceStyle",
        "dialogPreferenceStyle",
        "editTextPreferenceStyle",
        "ringtonePreferenceStyle",
        "preferenceLayoutChild",
        "textSize",
        "typeface",
        "textStyle",
        "textColor",
        "textColorHighlight",
        "textColorHint",
        "textColorLink",
        "state_focused",
        "state_window_focused",
        "state_enabled",
        "state_checkable",
        "state_checked",
        "state_selected",
        "state_active",
        "state_single",
        "state_first",
        "state_middle",
        "state_last",
        "state_pressed",
        "state_expanded",
        "state_empty",
        "state_above_anchor",
        "ellipsize",
        "x",
        "y",
        "windowAnimationStyle",
        "gravity",
        "autoLink",
        "linksClickable",
        "entries",
        "layout_gravity",
        "windowEnterAnimation",
        "windowExitAnimation",
        "windowShowAnimation",
        "windowHideAnimation",
        "activityOpenEnterAnimation",
        "activityOpenExitAnimation",
        "activityCloseEnterAnimation",
        "activityCloseExitAnimation",
        "taskOpenEnterAnimation",
        "taskOpenExitAnimation",
        "taskCloseEnterAnimation",
        "taskCloseExitAnimation",
        "taskToFrontEnterAnimation",
        "taskToFrontExitAnimation",
        "taskToBackEnterAnimation",
        "taskToBackExitAnimation",
        "orientation",
        "keycode",
        "fullDark",
        "topDark",
        "centerDark",
        "bottomDark",
        "fullBright",
        "topBright",
        "centerBright",
        "bottomBright",
        "bottomMedium",
        "centerMedium",
        "id",
        "tag",
        "scrollX",
        "scrollY",
        "background",
        "padding",
        "paddingLeft",
        "paddingTop",
        "paddingRight",
        "paddingBottom",
        "focusable",
        "focusableInTouchMode",
        "visibility",
        "fitsSystemWindows",
        "scrollbars",
        "fadingEdge",
        "fadingEdgeLength",
        "nextFocusLeft",
        "nextFocusRight",
        "nextFocusUp",
        "nextFocusDown",
        "clickable",
        "longClickable",
        "saveEnabled",
        "drawingCacheQuality",
        "duplicateParentState",
        "clipChildren",
        "clipToPadding",
        "layoutAnimation",
        "animationCache",
        "persistentDrawingCache",
        "alwaysDrawnWithCache",
        "addStatesFromChildren",
        "descendantFocusability",
        "layout",
        "inflatedId",
        "layout_width",
        "layout_height",
        "layout_margin",
        "layout_marginLeft",
        "layout_marginTop",
        "layout_marginRight",
        "layout_marginBottom",
        "listSelector",
        "drawSelectorOnTop",
        "stackFromBottom",
        "scrollingCache",
        "textFilterEnabled",
        "transcriptMode",
        "cacheColorHint",
        "dial",
        "hand_hour",
        "hand_minute",
        "format",
        "checked",
        "button",
        "checkMark",
        "foreground",
        "measureAllChildren",
        "groupIndicator",
        "childIndicator",
        "indicatorLeft",
        "indicatorRight",
        "childIndicatorLeft",
        "childIndicatorRight",
        "childDivider",
        "animationDuration",
        "spacing",
        "horizontalSpacing",
        "verticalSpacing",
        "stretchMode",
        "columnWidth",
        "numColumns",
        "src",
        "antialias",
        "filter",
        "dither",
        "scaleType",
        "adjustViewBounds",
        "maxWidth",
        "maxHeight",
        "tint",
        "baselineAlignBottom",
        "cropToPadding",
        "textOn",
        "textOff",
        "baselineAligned",
        "baselineAlignedChildIndex",
        "weightSum",
        "divider",
        "dividerHeight",
        "choiceMode",
        "itemTextAppearance",
        "horizontalDivider",
        "verticalDivider",
        "headerBackground",
        "itemBackground",
        "itemIconDisabledAlpha",
        "rowHeight",
        "maxRows",
        "maxItemsPerRow",
        "moreIcon",
        "max",
        "progress",
        "secondaryProgress",
        "indeterminate",
        "indeterminateOnly",
        "indeterminateDrawable",
        "progressDrawable",
        "indeterminateDuration",
        "indeterminateBehavior",
        "minWidth",
        "minHeight",
        "interpolator",
        "thumb",
        "thumbOffset",
        "numStars",
        "rating",
        "stepSize",
        "isIndicator",
        "checkedButton",
        "stretchColumns",
        "shrinkColumns",
        "collapseColumns",
        "layout_column",
        "layout_span",
        "bufferType",
        "text",
        "hint",
        "textScaleX",
        "cursorVisible",
        "maxLines",
        "lines",
        "height",
        "minLines",
        "maxEms",
        "ems",
        "width",
        "minEms",
        "scrollHorizontally",
        "password",
        "singleLine",
        "selectAllOnFocus",
        "includeFontPadding",
        "maxLength",
        "shadowColor",
        "shadowDx",
        "shadowDy",
        "shadowRadius",
        "numeric",
        "digits",
        "phoneNumber",
        "inputMethod",
        "capitalize",
        "autoText",
        "editable",
        "freezesText",
        "drawableTop",
        "drawableBottom",
        "drawableLeft",
        "drawableRight",
        "drawablePadding",
        "completionHint",
        "completionHintView",
        "completionThreshold",
        "dropDownSelector",
        "popupBackground",
        "inAnimation",
        "outAnimation",
        "flipInterval",
        "fillViewport",
        "prompt",
        "startYear",
        "endYear",
        "mode",
        "layout_x",
        "layout_y",
        "layout_weight",
        "layout_toLeftOf",
        "layout_toRightOf",
        "layout_above",
        "layout_below",
        "layout_alignBaseline",
        "layout_alignLeft",
        "layout_alignTop",
        "layout_alignRight",
        "layout_alignBottom",
        "layout_alignParentLeft",
        "layout_alignParentTop",
        "layout_alignParentRight",
        "layout_alignParentBottom",
        "layout_centerInParent",
        "layout_centerHorizontal",
        "layout_centerVertical",
        "layout_alignWithParentIfMissing",
        "layout_scale",
        "visible",
        "variablePadding",
        "constantSize",
        "oneshot",
        "duration",
        "drawable",
        "shape",
        "innerRadiusRatio",
        "thicknessRatio",
        "startColor",
        "endColor",
        "useLevel",
        "angle",
        "type",
        "centerX",
        "centerY",
        "gradientRadius",
        "color",
        "dashWidth",
        "dashGap",
        "radius",
        "topLeftRadius",
        "topRightRadius",
        "bottomLeftRadius",
        "bottomRightRadius",
        "left",
        "top",
        "right",
        "bottom",
        "minLevel",
        "maxLevel",
        "fromDegrees",
        "toDegrees",
        "pivotX",
        "pivotY",
        "insetLeft",
        "insetRight",
        "insetTop",
        "insetBottom",
        "shareInterpolator",
        "fillBefore",
        "fillAfter",
        "startOffset",
        "repeatCount",
        "repeatMode",
        "zAdjustment",
        "fromXScale",
        "toXScale",
        "fromYScale",
        "toYScale",
        "fromXDelta",
        "toXDelta",
        "fromYDelta",
        "toYDelta",
        "fromAlpha",
        "toAlpha",
        "delay",
        "animation",
        "animationOrder",
        "columnDelay",
        "rowDelay",
        "direction",
        "directionPriority",
        "factor",
        "cycles",
        "searchMode",
        "searchSuggestAuthority",
        "searchSuggestPath",
        "searchSuggestSelection",
        "searchSuggestIntentAction",
        "searchSuggestIntentData",
        "queryActionMsg",
        "suggestActionMsg",
        "suggestActionMsgColumn",
        "menuCategory",
        "orderInCategory",
        "checkableBehavior",
        "title",
        "titleCondensed",
        "alphabeticShortcut",
        "numericShortcut",
        "checkable",
        "selectable",
        "orderingFromXml",
        "key",
        "summary",
        "order",
        "widgetLayout",
        "dependency",
        "defaultValue",
        "shouldDisableView",
        "summaryOn",
        "summaryOff",
        "disableDependentsState",
        "dialogTitle",
        "dialogMessage",
        "dialogIcon",
        "positiveButtonText",
        "negativeButtonText",
        "dialogLayout",
        "entryValues",
        "ringtoneType",
        "showDefault",
        "showSilent",
        "scaleWidth",
        "scaleHeight",
        "scaleGravity",
        "ignoreGravity",
        "foregroundGravity",
        "tileMode",
        "targetActivity",
        "alwaysRetainTaskState",
        "allowTaskReparenting",
        "searchButtonText",
        "colorForegroundInverse",
        "textAppearanceButton",
        "listSeparatorTextViewStyle",
        "streamType",
        "clipOrientation",
        "centerColor",
        "minSdkVersion",
        "windowFullscreen",
        "unselectedAlpha",
        "progressBarStyleSmallTitle",
        "ratingBarStyleIndicator",
        "apiKey",
        "textColorTertiary",
        "textColorTertiaryInverse",
        "listDivider",
        "soundEffectsEnabled",
        "keepScreenOn",
        "lineSpacingExtra",
        "lineSpacingMultiplier",
        "listChoiceIndicatorSingle",
        "listChoiceIndicatorMultiple",
        "versionCode",
        "versionName",
        "marqueeRepeatLimit",
        "windowNoDisplay",
        "backgroundDimEnabled",
        "inputType",
        "isDefault",
        "windowDisablePreview",
        "privateImeOptions",
        "editorExtras",
        "settingsActivity",
        "fastScrollEnabled",
        "reqTouchScreen",
        "reqKeyboardType",
        "reqHardKeyboard",
        "reqNavigation",
        "windowSoftInputMode",
        "imeFullscreenBackground",
        "noHistory",
        "headerDividersEnabled",
        "footerDividersEnabled",
        "candidatesTextStyleSpans",
        "smoothScrollbar",
        "reqFiveWayNav",
        "keyBackground",
        "keyTextSize",
        "labelTextSize",
        "keyTextColor",
        "keyPreviewLayout",
        "keyPreviewOffset",
        "keyPreviewHeight",
        "verticalCorrection",
        "popupLayout",
        "state_long_pressable",
        "keyWidth",
        "keyHeight",
        "horizontalGap",
        "verticalGap",
        "rowEdgeFlags",
        "codes",
        "popupKeyboard",
        "popupCharacters",
        "keyEdgeFlags",
        "isModifier",
        "isSticky",
        "isRepeatable",
        "iconPreview",
        "keyOutputText",
        "keyLabel",
        "keyIcon",
        "keyboardMode",
        "isScrollContainer",
        "fillEnabled",
        "updatePeriodMillis",
        "initialLayout",
        "voiceSearchMode",
        "voiceLanguageModel",
        "voicePromptText",
        "voiceLanguage",
        "voiceMaxResults",
        "bottomOffset",
        "topOffset",
        "allowSingleTap",
        "handle",
        "content",
        "animateOnClick",
        "configure",
        "hapticFeedbackEnabled",
        "innerRadius",
        "thickness",
        "sharedUserLabel",
        "dropDownWidth",
        "dropDownAnchor",
        "imeOptions",
        "imeActionLabel",
        "imeActionId",
        "UNKNOWN",
        "imeExtractEnterAnimation",
        "imeExtractExitAnimation",
        "tension",
        "extraTension",
        "anyDensity",
        "searchSuggestThreshold",
        "includeInGlobalSearch",
        "onClick",
        "targetSdkVersion",
        "maxSdkVersion",
        "testOnly",
        "contentDescription",
        "gestureStrokeWidth",
        "gestureColor",
        "uncertainGestureColor",
        "fadeOffset",
        "fadeDuration",
        "gestureStrokeType",
        "gestureStrokeLengthThreshold",
        "gestureStrokeSquarenessThreshold",
        "gestureStrokeAngleThreshold",
        "eventsInterceptionEnabled",
        "fadeEnabled",
        "backupAgent",
        "allowBackup",
        "glEsVersion",
        "queryAfterZeroResults",
        "dropDownHeight",
        "smallScreens",
        "normalScreens",
        "largeScreens",
        "progressBarStyleInverse",
        "progressBarStyleSmallInverse",
        "progressBarStyleLargeInverse",
        "searchSettingsDescription",
        "textColorPrimaryInverseDisableOnly",
        "autoUrlDetect",
        "resizeable",
        "required",
        "accountType",
        "contentAuthority",
        "userVisible",
        "windowShowWallpaper",
        "wallpaperOpenEnterAnimation",
        "wallpaperOpenExitAnimation",
        "wallpaperCloseEnterAnimation",
        "wallpaperCloseExitAnimation",
        "wallpaperIntraOpenEnterAnimation",
        "wallpaperIntraOpenExitAnimation",
        "wallpaperIntraCloseEnterAnimation",
        "wallpaperIntraCloseExitAnimation",
        "supportsUploading",
        "killAfterRestore",
        "restoreNeedsApplication",
        "smallIcon",
        "accountPreferences",
        "textAppearanceSearchResultSubtitle",
        "textAppearanceSearchResultTitle",
        "summaryColumn",
        "detailColumn",
        "detailSocialSummary",
        "thumbnail",
        "detachWallpaper",
        "finishOnCloseSystemDialogs",
        "scrollbarFadeDuration",
        "scrollbarDefaultDelayBeforeFade",
        "fadeScrollbars",
        "colorBackgroundCacheHint",
        "dropDownHorizontalOffset",
        "dropDownVerticalOffset",
        "quickContactBadgeStyleWindowSmall",
        "quickContactBadgeStyleWindowMedium",
        "quickContactBadgeStyleWindowLarge",
        "quickContactBadgeStyleSmallWindowSmall",
        "quickContactBadgeStyleSmallWindowMedium",
        "quickContactBadgeStyleSmallWindowLarge",
        "author",
        "autoStart",
        "expandableListViewWhiteStyle",
        "installLocation",
        "vmSafeMode",
        "webTextViewStyle",
        "restoreAnyVersion",
        "tabStripLeft",
        "tabStripRight",
        "tabStripEnabled",
        "logo",
        "xlargeScreens",
        "immersive",
        "overScrollMode",
        "overScrollHeader",
        "overScrollFooter",
        "filterTouchesWhenObscured",
        "textSelectHandleLeft",
        "textSelectHandleRight",
        "textSelectHandle",
        "textSelectHandleWindowStyle",
        "popupAnimationStyle",
        "screenSize",
        "screenDensity",
        "allContactsName",
        "windowActionBar",
        "actionBarStyle",
        "navigationMode",
        "displayOptions",
        "subtitle",
        "customNavigationLayout",
        "hardwareAccelerated",
        "measureWithLargestChild",
        "animateFirstView",
        "dropDownSpinnerStyle",
        "actionDropDownStyle",
        "actionButtonStyle",
        "showAsAction",
        "previewImage",
        "actionModeBackground",
        "actionModeCloseDrawable",
        "windowActionModeOverlay",
        "valueFrom",
        "valueTo",
        "valueType",
        "propertyName",
        "ordering",
        "fragment",
        "windowActionBarOverlay",
        "fragmentOpenEnterAnimation",
        "fragmentOpenExitAnimation",
        "fragmentCloseEnterAnimation",
        "fragmentCloseExitAnimation",
        "fragmentFadeEnterAnimation",
        "fragmentFadeExitAnimation",
        "actionBarSize",
        "imeSubtypeLocale",
        "imeSubtypeMode",
        "imeSubtypeExtraValue",
        "splitMotionEvents",
        "listChoiceBackgroundIndicator",
        "spinnerMode",
        "animateLayoutChanges",
        "actionBarTabStyle",
        "actionBarTabBarStyle",
        "actionBarTabTextStyle",
        "actionOverflowButtonStyle",
        "actionModeCloseButtonStyle",
        "titleTextStyle",
        "subtitleTextStyle",
        "iconifiedByDefault",
        "actionLayout",
        "actionViewClass",
        "activatedBackgroundIndicator",
        "state_activated",
        "listPopupWindowStyle",
        "popupMenuStyle",
        "textAppearanceLargePopupMenu",
        "textAppearanceSmallPopupMenu",
        "breadCrumbTitle",
        "breadCrumbShortTitle",
        "listDividerAlertDialog",
        "textColorAlertDialogListItem",
        "loopViews",
        "dialogTheme",
        "alertDialogTheme",
        "dividerVertical",
        "homeAsUpIndicator",
        "enterFadeDuration",
        "exitFadeDuration",
        "selectableItemBackground",
        "autoAdvanceViewId",
        "useIntrinsicSizeAsMinimum",
        "actionModeCutDrawable",
        "actionModeCopyDrawable",
        "actionModePasteDrawable",
        "textEditPasteWindowLayout",
        "textEditNoPasteWindowLayout",
        "textIsSelectable",
        "windowEnableSplitTouch",
        "indeterminateProgressStyle",
        "progressBarPadding",
        "animationResolution",
        "state_accelerated",
        "baseline",
        "homeLayout",
        "opacity",
        "alpha",
        "transformPivotX",
        "transformPivotY",
        "translationX",
        "translationY",
        "scaleX",
        "scaleY",
        "rotation",
        "rotationX",
        "rotationY",
        "showDividers",
        "dividerPadding",
        "borderlessButtonStyle",
        "dividerHorizontal",
        "itemPadding",
        "buttonBarStyle",
        "buttonBarButtonStyle",
        "segmentedButtonStyle",
        "staticWallpaperPreview",
        "allowParallelSyncs",
        "isAlwaysSyncable",
        "verticalScrollbarPosition",
        "fastScrollAlwaysVisible",
        "fastScrollThumbDrawable",
        "fastScrollPreviewBackgroundLeft",
        "fastScrollPreviewBackgroundRight",
        "fastScrollTrackDrawable",
        "fastScrollOverlayPosition",
        "customTokens",
        "nextFocusForward",
        "firstDayOfWeek",
        "showWeekNumber",
        "minDate",
        "maxDate",
        "shownWeekCount",
        "selectedWeekBackgroundColor",
        "focusedMonthDateColor",
        "unfocusedMonthDateColor",
        "weekNumberColor",
        "weekSeparatorLineColor",
        "selectedDateVerticalBar",
        "weekDayTextAppearance",
        "dateTextAppearance",
        "UNKNOWN",
        "spinnersShown",
        "calendarViewShown",
        "state_multiline",
        "detailsElementBackground",
        "textColorHighlightInverse",
        "textColorLinkInverse",
        "editTextColor",
        "editTextBackground",
        "horizontalScrollViewStyle",
        "layerType",
        "alertDialogIcon",
        "windowMinWidthMajor",
        "windowMinWidthMinor",
        "queryHint",
        "fastScrollTextColor",
        "largeHeap",
        "windowCloseOnTouchOutside",
        "datePickerStyle",
        "calendarViewStyle",
        "textEditSidePasteWindowLayout",
        "textEditSideNoPasteWindowLayout",
        "actionMenuTextAppearance",
        "actionMenuTextColor",
        "textCursorDrawable",
        "resizeMode",
        "requiresSmallestWidthDp",
        "compatibleWidthLimitDp",
        "largestWidthLimitDp",
        "state_hovered",
        "state_drag_can_accept",
        "state_drag_hovered",
        "stopWithTask",
        "switchTextOn",
        "switchTextOff",
        "switchPreferenceStyle",
        "switchTextAppearance",
        "track",
        "switchMinWidth",
        "switchPadding",
        "thumbTextPadding",
        "textSuggestionsWindowStyle",
        "textEditSuggestionItemLayout",
        "rowCount",
        "rowOrderPreserved",
        "columnCount",
        "columnOrderPreserved",
        "useDefaultMargins",
        "alignmentMode",
        "layout_row",
        "layout_rowSpan",
        "layout_columnSpan",
        "actionModeSelectAllDrawable",
        "isAuxiliary",
        "accessibilityEventTypes",
        "packageNames",
        "accessibilityFeedbackType",
        "notificationTimeout",
        "accessibilityFlags",
        "canRetrieveWindowContent",
        "listPreferredItemHeightLarge",
        "listPreferredItemHeightSmall",
        "actionBarSplitStyle",
        "actionProviderClass",
        "backgroundStacked",
        "backgroundSplit",
        "textAllCaps",
        "colorPressedHighlight",
        "colorLongPressedHighlight",
        "colorFocusedHighlight",
        "colorActivatedHighlight",
        "colorMultiSelectHighlight",
        "drawableStart",
        "drawableEnd",
        "actionModeStyle",
        "minResizeWidth",
        "minResizeHeight",
        "actionBarWidgetTheme",
        "uiOptions",
        "subtypeLocale",
        "subtypeExtraValue",
        "actionBarDivider",
        "actionBarItemBackground",
        "actionModeSplitBackground",
        "textAppearanceListItem",
        "textAppearanceListItemSmall",
        "targetDescriptions",
        "directionDescriptions",
        "overridesImplicitlyEnabledSubtype",
        "listPreferredItemPaddingLeft",
        "listPreferredItemPaddingRight",
        "requiresFadingEdge",
        "publicKey",
        "parentActivityName",
        "UNKNOWN",
        "isolatedProcess",
        "importantForAccessibility",
        "keyboardLayout",
        "fontFamily",
        "mediaRouteButtonStyle",
        "mediaRouteTypes",
        "supportsRtl",
        "textDirection",
        "textAlignment",
        "layoutDirection",
        "paddingStart",
        "paddingEnd",
        "layout_marginStart",
        "layout_marginEnd",
        "layout_toStartOf",
        "layout_toEndOf",
        "layout_alignStart",
        "layout_alignEnd",
        "layout_alignParentStart",
        "layout_alignParentEnd",
        "listPreferredItemPaddingStart",
        "listPreferredItemPaddingEnd",
        "singleUser",
        "presentationTheme",
        "subtypeId",
        "initialKeyguardLayout",
        "UNKNOWN",
        "widgetCategory",
        "permissionGroupFlags",
        "labelFor",
        "permissionFlags",
        "checkedTextViewStyle",
        "showOnLockScreen",
        "format12Hour",
        "format24Hour",
        "timeZone",
        "mipMap",
        "mirrorForRtl",
        "windowOverscan",
        "requiredForAllUsers",
        "indicatorStart",
        "indicatorEnd",
        "childIndicatorStart",
        "childIndicatorEnd",
        "restrictedAccountType",
        "requiredAccountType",
        "canRequestTouchExplorationMode",
        "canRequestEnhancedWebAccessibility",
        "canRequestFilterKeyEvents",
        "layoutMode",
        "keySet",
        "targetId",
        "fromScene",
        "toScene",
        "transition",
        "transitionOrdering",
        "fadingMode",
        "startDelay",
        "ssp",
        "sspPrefix",
        "sspPattern",
        "addPrintersActivity",
        "vendor",
        "category",
        "isAsciiCapable",
        "autoMirrored",
        "supportsSwitchingToNextInputMethod",
        "requireDeviceUnlock",
        "apduServiceBanner",
        "accessibilityLiveRegion",
        "windowTranslucentStatus",
        "windowTranslucentNavigation",
        "advancedPrintOptionsActivity",
        "banner",
        "windowSwipeToDismiss",
        "isGame",
        "allowEmbedded",
        "setupActivity",
        "fastScrollStyle",
        "windowContentTransitions",
        "windowContentTransitionManager",
        "translationZ",
        "tintMode",
        "controlX1",
        "controlY1",
        "controlX2",
        "controlY2",
        "transitionName",
        "transitionGroup",
        "viewportWidth",
        "viewportHeight",
        "fillColor",
        "pathData",
        "strokeColor",
        "strokeWidth",
        "trimPathStart",
        "trimPathEnd",
        "trimPathOffset",
        "strokeLineCap",
        "strokeLineJoin",
        "strokeMiterLimit",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "colorControlNormal",
        "colorControlActivated",
        "colorButtonNormal",
        "colorControlHighlight",
        "persistableMode",
        "titleTextAppearance",
        "subtitleTextAppearance",
        "slideEdge",
        "actionBarTheme",
        "textAppearanceListItemSecondary",
        "colorPrimary",
        "colorPrimaryDark",
        "colorAccent",
        "nestedScrollingEnabled",
        "windowEnterTransition",
        "windowExitTransition",
        "windowSharedElementEnterTransition",
        "windowSharedElementExitTransition",
        "windowAllowReturnTransitionOverlap",
        "windowAllowEnterTransitionOverlap",
        "sessionService",
        "stackViewStyle",
        "switchStyle",
        "elevation",
        "excludeId",
        "excludeClass",
        "hideOnContentScroll",
        "actionOverflowMenuStyle",
        "documentLaunchMode",
        "maxRecents",
        "autoRemoveFromRecents",
        "stateListAnimator",
        "toId",
        "fromId",
        "reversible",
        "splitTrack",
        "targetName",
        "excludeName",
        "matchOrder",
        "windowDrawsSystemBarBackgrounds",
        "statusBarColor",
        "navigationBarColor",
        "contentInsetStart",
        "contentInsetEnd",
        "contentInsetLeft",
        "contentInsetRight",
        "paddingMode",
        "layout_rowWeight",
        "layout_columnWeight",
        "translateX",
        "translateY",
        "selectableItemBackgroundBorderless",
        "elegantTextHeight",
        "UNKNOWN",
        "UNKNOWN",
        "UNKNOWN",
        "windowTransitionBackgroundFadeDuration",
        "overlapAnchor",
        "progressTint",
        "progressTintMode",
        "progressBackgroundTint",
        "progressBackgroundTintMode",
        "secondaryProgressTint",
        "secondaryProgressTintMode",
        "indeterminateTint",
        "indeterminateTintMode",
        "backgroundTint",
        "backgroundTintMode",
        "foregroundTint",
        "foregroundTintMode",
        "buttonTint",
        "buttonTintMode",
        "thumbTint",
        "thumbTintMode",
        "fullBackupOnly",
        "propertyXName",
        "propertyYName",
        "relinquishTaskIdentity",
        "tileModeX",
        "tileModeY",
        "actionModeShareDrawable",
        "actionModeFindDrawable",
        "actionModeWebSearchDrawable",
        "transitionVisibilityMode",
        "minimumHorizontalAngle",
        "minimumVerticalAngle",
        "maximumAngle",
        "searchViewStyle",
        "closeIcon",
        "goIcon",
        "searchIcon",
        "voiceIcon",
        "commitIcon",
        "suggestionRowLayout",
        "queryBackground",
        "submitBackground",
        "buttonBarPositiveButtonStyle",
        "buttonBarNeutralButtonStyle",
        "buttonBarNegativeButtonStyle",
        "popupElevation",
        "actionBarPopupTheme",
        "multiArch",
        "touchscreenBlocksFocus",
        "windowElevation",
        "launchTaskBehindTargetAnimation",
        "launchTaskBehindSourceAnimation",
        "restrictionType",
        "dayOfWeekBackground",
        "dayOfWeekTextAppearance",
        "headerMonthTextAppearance",
        "headerDayOfMonthTextAppearance",
        "headerYearTextAppearance",
        "yearListItemTextAppearance",
        "yearListSelectorColor",
        "calendarTextColor",
        "recognitionService",
        "timePickerStyle",
        "timePickerDialogTheme",
        "headerTimeTextAppearance",
        "headerAmPmTextAppearance",
        "numbersTextColor",
        "numbersBackgroundColor",
        "numbersSelectorColor",
        "amPmTextColor",
        "amPmBackgroundColor",
        "UNKNOWN",
        "checkMarkTint",
        "checkMarkTintMode",
        "popupTheme",
        "toolbarStyle",
        "windowClipToOutline",
        "datePickerDialogTheme",
        "showText",
        "windowReturnTransition",
        "windowReenterTransition",
        "windowSharedElementReturnTransition",
        "windowSharedElementReenterTransition",
        "resumeWhilePausing",
        "datePickerMode",
        "timePickerMode",
        "inset",
        "letterSpacing",
        "fontFeatureSettings",
        "outlineProvider",
        "contentAgeHint",
        "country",
        "windowSharedElementsUseOverlay",
        "reparent",
        "reparentWithOverlay",
        "ambientShadowAlpha",
        "spotShadowAlpha",
        "navigationIcon",
        "navigationContentDescription",
        "fragmentExitTransition",
        "fragmentEnterTransition",
        "fragmentSharedElementEnterTransition",
        "fragmentReturnTransition",
        "fragmentSharedElementReturnTransition",
        "fragmentReenterTransition",
        "fragmentAllowEnterTransitionOverlap",
        "fragmentAllowReturnTransitionOverlap",
        "patternPathData",
        "strokeAlpha",
        "fillAlpha",
        "windowActivityTransitions",
        "colorEdgeEffect",
        "resizeClip",
        "collapseContentDescription",
        "accessibilityTraversalBefore",
        "accessibilityTraversalAfter",
        "dialogPreferredPadding",
        "searchHintIcon",
        "revisionCode",
        "drawableTint",
        "drawableTintMode",
        "fraction",
        "trackTint",
        "trackTintMode",
        "start",
        "end",
        "breakStrategy",
        "hyphenationFrequency",
        "allowUndo",
        "windowLightStatusBar",
        "numbersInnerTextColor",
        "colorBackgroundFloating",
        "titleTextColor",
        "subtitleTextColor",
        "thumbPosition",
        "scrollIndicators",
        "contextClickable",
        "fingerprintAuthDrawable",
        "logoDescription",
        "extractNativeLibs",
        "fullBackupContent",
        "usesCleartextTraffic",
        "lockTaskMode",
        "autoVerify",
        "showForAllUsers",
        "supportsAssist",
        "supportsLaunchVoiceAssistFromKeyguard",
        "listMenuViewStyle",
        "subMenuArrow",
        "defaultWidth",
        "defaultHeight",
        "resizeableActivity",
        "supportsPictureInPicture",
        "titleMargin",
        "titleMarginStart",
        "titleMarginEnd",
        "titleMarginTop",
        "titleMarginBottom",
        "maxButtonHeight",
        "buttonGravity",
        "collapseIcon",
        "level",
        "contextPopupMenuStyle",
        "textAppearancePopupMenuHeader",
        "windowBackgroundFallback",
        "defaultToDeviceProtectedStorage",
        "directBootAware",
        "preferenceFragmentStyle",
        "canControlMagnification",
        "languageTag",
        "pointerIcon",
        "tickMark",
        "tickMarkTint",
        "tickMarkTintMode",
        "canPerformGestures",
        "externalService",
        "supportsLocalInteraction",
        "startX",
        "startY",
        "endX",
        "endY",
        "offset",
        "use32bitAbi",
        "bitmap",
        "hotSpotX",
        "hotSpotY",
        "version",
        "backupInForeground",
        "countDown",
        "canRecord",
        "tunerCount",
        "fillType",
        "popupEnterTransition",
        "popupExitTransition",
        "forceHasOverlappingRendering",
        "contentInsetStartWithNavigation",
        "contentInsetEndWithActions",
        "numberPickerStyle",
        "enableVrMode",
        "UNKNOWN",
        "networkSecurityConfig",
        "shortcutId",
        "shortcutShortLabel",
        "shortcutLongLabel",
        "shortcutDisabledMessage",
        "roundIcon",
        "contextUri",
        "contextDescription",
        "showMetadataInPreview",
        "colorSecondary",
    ];

    let i = resource_id - 0x1010000;

    Some(
        RESOURCE_STRINGS
            .get(usize::try_from(i).unwrap())?
            .to_string(),
    )
}
