use std::path::Path;

use anyhow::{anyhow, Context, Result};
use async_trait::async_trait;
use clap::Parser;
use log::info;
use tokio::fs;

use crate::{
    compile::build_awto_pkg,
    util::{add_package_to_workspace, CargoFile},
    Runnable,
};

use super::prepare_awto_dir;

/// Compiles database package from app schema
#[derive(Parser)]
pub struct Database {
    /// Prints more information
    #[clap(short, long)]
    pub verbose: bool,
}

#[async_trait]
impl Runnable for Database {
    async fn run(&mut self) -> Result<()> {
        let cargo_file = CargoFile::load("./schema/Cargo.toml")
            .await
            .context("could not load schema Cargo.toml file from './schema/Cargo.toml'")?;
        if cargo_file
            .package
            .as_ref()
            .map(|package| package.name != "schema")
            .unwrap_or(false)
        {
            match cargo_file.package {
                Some(package) => {
                    return Err(anyhow!(
                        "schema package must be named 'schema' but is named '{}'",
                        package.name
                    ));
                }
                None => return Err(anyhow!("schema package must be named 'schema'")),
            }
        }

        prepare_awto_dir().await?;

        Self::prepare_database_dir().await?;
        add_package_to_workspace("awto/database").await?;
        build_awto_pkg("database").await?;

        info!("compiled package 'database'");

        Ok(())
    }

    fn is_verbose(&self) -> bool {
        self.verbose
    }
}

impl Database {
    const DATABASE_DIR: &'static str = "./awto/database";
    const DATABASE_SRC_DIR: &'static str = "./awto/database/src";
    const DATABASE_CARGO_PATH: &'static str = "./awto/database/Cargo.toml";
    const DATABASE_CARGO_TOML_BYTES: &'static [u8] =
        include_bytes!("../templates/database/Cargo.toml.template");
    const DATABASE_BUILD_PATH: &'static str = "./awto/database/build.rs";
    const DATABASE_BUILD_BYTES: &'static [u8] =
        include_bytes!("../templates/database/build.rs.template");
    const DATABASE_LIB_PATH: &'static str = "./awto/database/src/lib.rs";

    async fn prepare_database_dir() -> Result<()> {
        if Path::new(Self::DATABASE_DIR).is_dir() {
            fs::remove_dir_all(Self::DATABASE_DIR)
                .await
                .with_context(|| format!("could not delete directory '{}'", Self::DATABASE_DIR))?;
        }

        fs::create_dir(Self::DATABASE_DIR)
            .await
            .with_context(|| format!("could not create directory '{}'", Self::DATABASE_DIR))?;

        fs::create_dir(Self::DATABASE_SRC_DIR)
            .await
            .with_context(|| format!("could not create directory '{}'", Self::DATABASE_SRC_DIR))?;

        fs::write(Self::DATABASE_CARGO_PATH, Self::DATABASE_CARGO_TOML_BYTES)
            .await
            .with_context(|| format!("could not write file '{}'", Self::DATABASE_CARGO_PATH))?;

        fs::write(Self::DATABASE_BUILD_PATH, Self::DATABASE_BUILD_BYTES)
            .await
            .with_context(|| format!("could not write file '{}'", Self::DATABASE_BUILD_PATH))?;

        let lib_content = concat!(
            "// This file is automatically @generated by ",
            env!("CARGO_PKG_NAME"),
            " v",
            env!("CARGO_PKG_VERSION"),
            "\n\npub use sea_orm;\n\ninclude!(concat!(env!(\"OUT_DIR\"), \"/app.rs\"));\n"
        )
        .to_string();

        fs::write(Self::DATABASE_LIB_PATH, lib_content)
            .await
            .with_context(|| format!("could not write file '{}'", Self::DATABASE_LIB_PATH))?;

        Ok(())
    }
}
