/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

//! Credential provider augmentation through the AWS Security Token Service (STS).

mod assume_role;
pub use assume_role::{AssumeRoleProvider, AssumeRoleProviderBuilder};

pub(crate) mod util {
    use aws_sdk_sts::model::Credentials as StsCredentials;
    use aws_types::credentials::{self, CredentialsError};
    use aws_types::Credentials as AwsCredentials;
    use std::time::{SystemTime, UNIX_EPOCH};

    /// Convert STS credentials to aws_auth::Credentials
    pub(crate) fn into_credentials(
        sts_credentials: Option<StsCredentials>,
        provider_name: &'static str,
    ) -> credentials::Result {
        let sts_credentials = sts_credentials
            .ok_or_else(|| CredentialsError::unhandled("STS credentials must be defined"))?;
        let expiration = sts_credentials
            .expiration
            .ok_or_else(|| CredentialsError::unhandled("missing expiration"))?;
        let expiration = expiration.to_system_time().ok_or_else(|| {
            CredentialsError::unhandled(format!(
                "expiration is before unix epoch: {:?}",
                &expiration
            ))
        })?;
        Ok(AwsCredentials::new(
            sts_credentials
                .access_key_id
                .ok_or_else(|| CredentialsError::unhandled("access key id missing from result"))?,
            sts_credentials
                .secret_access_key
                .ok_or_else(|| CredentialsError::unhandled("secret access token missing"))?,
            sts_credentials.session_token,
            Some(expiration),
            provider_name,
        ))
    }

    /// Create a default STS session name
    ///
    /// STS Assume Role providers MUST assign a name to their generated session. When a user does not
    /// provide a name for the session, the provider will choose a name composed of a base + a timestamp,
    /// eg. `profile-file-provider-123456789`
    pub(crate) fn default_session_name(base: &str) -> String {
        let now = SystemTime::now()
            .duration_since(UNIX_EPOCH)
            .expect("post epoch");
        format!("{}-{}", base, now.as_millis())
    }
}
