//! This crate provides a convenient derive macro for `actix_web::ResponseError` trait.
//!
//! # Example
//! ## With Enum
//! ```
//! use awred::ResponseError;
//! use serde::Serialize;
//! use thiserror::Error;
//!
//! #[derive(Debug, Error, ResponseError, Serialize)]
//! pub enum AnError {
//!     #[error("Requested resource was not found")]
//!     #[status_code(NOT_FOUND)]
//!     ResourceNotFound,
//!
//!     #[error("Forbidden: {reason}")]
//!     #[status_code(FORBIDDEN)]
//!     Forbidden { reason: String },
//!
//!     // Internal Server Error
//!     #[error(transparent)]
//!     #[serde(skip)]
//!     IoError(#[from] std::io::Error),
//! }
//! ```
//!
//! ## With Struct
//! ```
//! # use awred::ResponseError;
//! # use serde::Serialize;
//! # use thiserror::Error;
//! #
//! #[derive(Debug, Error, ResponseError, Serialize)]
//! #[error("Invalid username or password")]
//! #[status_code(BAD_REQUEST)]
//! pub struct InvalidCredentials;
//! ```
//!
//! # Details
//! - Status codes (from `actix_web::http::StatusCode`) are specified with `#[status_code(...)]` attribute
//! - Variants/structs without `#[status_code(...)]` attribute return Internal Server Error with empty body
//! - Response body consists of serialised error and message (`error.to_string()`)
//!
//! # Error response body format
//! ```json
//! {
//!     "error": error,
//!     "message": error.to_string(),
//! }
//! ```

#![forbid(unsafe_code, clippy::unwrap_used)]
use proc_macro2::TokenStream;
use quote::quote;

/// Derive `ResponseError` trait
#[proc_macro_derive(ResponseError, attributes(status_code))]
pub fn derive_response_error(input: proc_macro::TokenStream) -> proc_macro::TokenStream {
    derive(syn::parse_macro_input!(input))
        .unwrap_or_else(|e| e.to_compile_error())
        .into()
}

/// Generate implementation
fn derive(input: syn::DeriveInput) -> syn::Result<TokenStream> {
    let ident = input.ident;

    match input.data {
        syn::Data::Enum(e) => {
            let response_variants = e
                .variants
                .into_iter()
                .filter_map(ResponseVariant::from_variant)
                .collect::<Result<Vec<_>, _>>()?;

            let status_code_arms = response_variants
                .iter()
                .map(ResponseVariant::to_status_code_arm);

            let error_response_patterns = response_variants
                .iter()
                .map(ResponseVariant::to_error_response_pattern);

            let error_response_arm = if response_variants.len() != 0 {
                quote! {
                    #(#error_response_patterns)|* => {
                        ::actix_web::HttpResponse::build(self.status_code()).json(::serde_json::json!({
                            "error": self,
                            "message": self.to_string(),
                        }))
                    }
                }
            } else {
                quote! {}
            };

            Ok(quote! {
                impl ::actix_web::ResponseError for #ident {
                    fn status_code(&self) -> ::actix_web::http::StatusCode {
                        match self {
                            #(#status_code_arms,)*
                            _ => ::actix_web::http::StatusCode::INTERNAL_SERVER_ERROR,
                        }
                    }

                    fn error_response(&self) -> ::actix_web::HttpResponse {
                        match self {
                            #error_response_arm
                            _ => ::actix_web::HttpResponse::InternalServerError().finish(),
                        }
                    }
                }
            })
        }
        syn::Data::Struct(_) => match get_status_code(&input.attrs) {
            Some(Ok(status_code)) => Ok(quote! {
                impl ::actix_web::ResponseError for #ident {
                    fn status_code(&self) -> ::actix_web::http::StatusCode {
                        ::actix_web::http::StatusCode::#status_code
                    }

                    fn error_response(&self) -> ::actix_web::HttpResponse {
                        ::actix_web::HttpResponse::build(self.status_code()).json(::serde_json::json!({
                            "error": self,
                            "message": self.to_string(),
                        }))
                    }
                }
            }),
            None => Ok(quote! {
                impl ::actix_web::ResponseError for #ident {
                    fn error_response(&self) -> ::actix_web::HttpResponse {
                        ::actix_web::HttpResponse::InternalServerError().finish()
                    }
                }
            }),
            Some(Err(e)) => Err(e),
        },
        syn::Data::Union(_) => Err(syn::Error::new_spanned(
            ident,
            "ResponseError derive cannot be applied to unions",
        )),
    }
}

/// Parse `#[status_code(...)]` attribute to status code identifier
fn get_status_code(attrs: &[syn::Attribute]) -> Option<syn::Result<syn::Ident>> {
    let response_attrs: Vec<_> = attrs
        .iter()
        .filter(|attr| attr.path.is_ident("status_code"))
        .collect();

    match response_attrs.len() {
        1 => Some(response_attrs[0].parse_args()),
        0 => None,
        _ => Some(Err(syn::Error::new_spanned(
            response_attrs[1],
            "only one #[status_code(...)] attribute is allowed",
        ))),
    }
}

/// An error enum variant with `#[status_code(...)]` attribute
struct ResponseVariant {
    pub status_code: syn::Ident,
    pub variant: syn::Ident,
}

impl ResponseVariant {
    /// Parse enum variant with `#[status_code(...)]` attribute to `ResponseVariant`
    pub fn from_variant(variant: syn::Variant) -> Option<syn::Result<Self>> {
        let ident = variant.ident;

        get_status_code(&variant.attrs).map(|r| {
            r.map(|status_code| Self {
                status_code,
                variant: ident,
            })
        })
    }

    /// Generate match arm for status code
    pub fn to_status_code_arm(&self) -> TokenStream {
        let Self {
            status_code,
            variant,
        } = self;

        quote! {
            Self::#variant { .. } => ::actix_web::http::StatusCode::#status_code
        }
    }

    /// Generate a match pattern for error response
    pub fn to_error_response_pattern(&self) -> TokenStream {
        let variant = &self.variant;

        quote! {
            Self::#variant { .. }
        }
    }
}
