//! This crate provides a convenient derive macro for `actix_web::ResponseError` trait.
//!
//! # Example
//! ```
//! use awred::ResponseError;
//! use serde::Serialize;
//! use thiserror::Error;
//!
//! #[derive(Debug, Error, ResponseError, Serialize)]
//! pub enum AnError {
//!     #[error("Requested resource was not found")]
//!     #[response(NOT_FOUND)]
//!     ResourceNotFound,
//!
//!     #[error("Forbidden: {reason}")]
//!     #[response(FORBIDDEN)]
//!     Forbidden { reason: String },
//!
//!     // Internal Server Error
//!     #[error(transparent)]
//!     #[serde(skip)]
//!     IoError(#[from] std::io::Error),
//! }
//! ```
//!
//! # Details
//! - Status codes (from `actix_web::http::StatusCode`) are specified in `#[response(...)]` attribute
//! - Variants without `#[response(...)]` attribute return Internal Server Error with empty body
//! - Response body consists of serialised error and message (`error.to_string()`)
//!
//! # Error response body format
//! ```json
//! {
//!     "error": error,
//!     "message": error.to_string(),
//! }
//! ```

#![forbid(unsafe_code, clippy::unwrap_used)]
use proc_macro2::TokenStream;
use quote::quote;

/// Derive `ResponseError` trait
#[proc_macro_derive(ResponseError, attributes(response))]
pub fn derive_response_error(input: proc_macro::TokenStream) -> proc_macro::TokenStream {
    derive(syn::parse_macro_input!(input))
        .unwrap_or_else(|e| e.to_compile_error())
        .into()
}

/// Generate implementation
fn derive(input: syn::DeriveInput) -> syn::Result<TokenStream> {
    let ident = input.ident;

    if let syn::Data::Enum(e) = input.data {
        let response_variants = e
            .variants
            .into_iter()
            .filter_map(ResponseVariant::from_variant)
            .collect::<Result<Vec<_>, _>>()?;

        let status_code_arms = response_variants
            .iter()
            .map(ResponseVariant::to_status_code_arm);

        let error_response_patterns = response_variants
            .iter()
            .map(ResponseVariant::to_error_response_pattern);

        let error_response_arm = if response_variants.len() != 0 {
            quote! {
                #(#error_response_patterns)|* => {
                    ::actix_web::HttpResponse::build(self.status_code()).json(::serde_json::json!({
                        "error": self,
                        "message": self.to_string(),
                    }))
                }
            }
        } else {
            quote! {}
        };

        Ok(quote! {
            impl ::actix_web::ResponseError for #ident {
                fn status_code(&self) -> ::actix_web::http::StatusCode {
                    match self {
                        #(#status_code_arms,)*
                        _ => ::actix_web::http::StatusCode::INTERNAL_SERVER_ERROR,
                    }
                }

                fn error_response(&self) -> ::actix_web::HttpResponse {
                    match self {
                        #error_response_arm
                        _ => ::actix_web::HttpResponse::InternalServerError().finish(),
                    }
                }
            }
        })
    } else {
        Err(syn::Error::new_spanned(
            ident,
            "ResponseError derive can only be applied to enums",
        ))
    }
}

/// An error enum variant with `#[response(...)]` attribute
struct ResponseVariant {
    pub status_code: syn::Ident,
    pub variant: syn::Ident,
}

impl ResponseVariant {
    /// Parse enum variant with `#[response(...)]` attribute to `ResponseVariant`
    pub fn from_variant(variant: syn::Variant) -> Option<syn::Result<Self>> {
        let attrs: Vec<_> = variant
            .attrs
            .iter()
            .filter(|attr| attr.path.is_ident("response"))
            .collect();

        match attrs.len() {
            1 => Some(attrs[0].parse_args().map(|status_code| Self {
                status_code,
                variant: variant.ident,
            })),
            0 => None,
            _ => Some(Err(syn::Error::new_spanned(
                attrs[1],
                "only one #[response(...)] attribute is allowed",
            ))),
        }
    }

    /// Generate match arm for status code
    pub fn to_status_code_arm(&self) -> TokenStream {
        let Self {
            status_code,
            variant,
        } = self;

        quote! {
            Self::#variant { .. } => ::actix_web::http::StatusCode::#status_code
        }
    }

    /// Generate a match pattern for error response
    pub fn to_error_response_pattern(&self) -> TokenStream {
        let variant = &self.variant;

        quote! {
            Self::#variant { .. }
        }
    }
}
