/* generic.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Generic serial EEPROM implementations


// Imports ===================================================================
use core::cell::RefCell;
use avr_oxide::devices::serialbus::SerialBusClient;
use avrox_storage::{RandomRead, RandomWrite};

// Declarations ==============================================================
/// 8 bit wide data, 16 bit wide address implementation, with big-endian
/// addressing.
pub struct SerPromD8A16be<const PROM_MAX: u16, const PAGESIZE: u16, BC: SerialBusClient> {
  bus: RefCell<BC>
}

// Code ======================================================================
impl<const PROM_MAX:u16, const PAGESIZE:u16, BC> SerPromD8A16be<PROM_MAX,PAGESIZE,BC>
where
  BC: SerialBusClient
{
  pub fn using_client(client: BC) -> Self {
    SerPromD8A16be {
      bus: RefCell::new(client)
    }
  }

  fn reduce_len_to_page_boundary(&self, addr: u16, len: u16) -> u16 {
    let page_max = (((addr / PAGESIZE) + 1) * PAGESIZE)-1;
    let max_len = page_max - addr;

    if max_len > len {
      len
    } else {
      max_len
    }
  }
}

impl<const PROM_MAX:u16, const PAGESIZE:u16, BC> RandomRead<u16> for SerPromD8A16be<PROM_MAX,PAGESIZE,BC>
  where
    BC: SerialBusClient
{
  fn read_at(&self, addr: u16, buf: &mut [u8]) -> avr_oxide::io::Result<usize> {
    if addr <= PROM_MAX {
      let max_bytes = self.reduce_len_to_page_boundary(addr, buf.len() as u16);

      let data_buf = &mut buf[0..max_bytes as usize];

      let addr_buf = [
        (addr >> 8) as u8,
        addr as u8
      ];

      match self.bus.borrow_mut().write_then_read(&addr_buf, data_buf) {
        Ok(bytes) => {
          Ok(bytes)
        },
        Err(_) => {
          Err(avr_oxide::io::IoError::Unknown)
        }
      }
    } else {
      Err(avr_oxide::io::IoError::EndOfFile)
    }
  }
}

impl<const PROM_MAX:u16, const PAGESIZE:u16, BC> RandomWrite<u16> for SerPromD8A16be<PROM_MAX,PAGESIZE,BC>
  where
    BC: SerialBusClient
{
  fn write_all_at(&mut self, addr: u16, buf: &[u8]) -> avr_oxide::io::Result<()> {
    let mut start = 0usize;

    while start < buf.len() {
      match self.write_at(addr + start as u16, &buf[start..]) {
        Ok(bytes) => {
          start += bytes;
        },
        Err(e) => {
          return Err(e)
        }
      }
    }
    Ok(())
  }


  fn write_at(&mut self, addr: u16, buf: &[u8]) -> avr_oxide::io::Result<usize> {
    if addr < PROM_MAX {
      let max_bytes = self.reduce_len_to_page_boundary(addr, buf.len() as u16);

      let command_buf = [
        &[
          (addr >> 8) as u8,
          addr as u8
        ],
        &buf[0..max_bytes as usize]
      ];

      match self.bus.borrow_mut().write_from_multiple(&command_buf) {
        Ok(_) => {
          Ok(max_bytes as usize)
        },
        Err(_) => {
          Err(avr_oxide::io::IoError::Unknown)
        }
      }
    } else {
      Err(avr_oxide::io::IoError::NoFreeSpace)
    }
  }

  fn flush(&mut self) -> avr_oxide::io::Result<()> {
    // We don't have buffers to flush, so this will always succeed
    Ok(())
  }
}

// Tests =====================================================================
