/* lib.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */

//! Storage device drivers for the [AVRoxide] operating system.
//!
//! [AVRoxide]: https://avroxi.de/

#![cfg_attr(target_arch="avr", no_std)]

extern crate self as avrox_storage;


pub mod serprom;

pub trait RandomRead<ADDR> {
  /// Read into the given buffer, from the given device address.
  ///
  /// If succesful, the number of bytes actually read will be returned in
  /// the Ok() result.
  ///
  #[must_use = "the buffer is not guaranteed to be filled; the Result will indicate how many bytes were actually read"]
  fn read_at(&self, addr: ADDR, buf: &mut [u8]) -> avr_oxide::io::Result<usize>;
}

pub trait RandomWrite<ADDR> {
  /// Write the contents of the given buffer at the given device address.
  /// THe entire buffer will be written unless an error condition occurs.
  /// If an error does occur, the state (partially written, not written at all)
  /// is undefined.
  fn write_all_at(&mut self, addr: ADDR, buf: &[u8]) -> avr_oxide::io::Result<()>;

  /// Write the contents of the given buffer at the given device address.
  /// Depending on the nature of the underlying device, the write may not
  /// complete.  The number of bytes actually written will be returned in the
  /// Ok() result, and it is the caller's responsibility to resume writing
  /// the remainder.
  #[must_use = "the entire buffer is not guaranteed to be written; the Result will indicate how many bytes were actually written"]
  fn write_at(&mut self, addr: ADDR, buf: &[u8]) -> avr_oxide::io::Result<usize>;

  /// Flush the write to the device.  Writes are not guaranteed to have been
  /// completed unless this method has been called.
  fn flush(&mut self) -> avr_oxide::io::Result<()>;
}